package iip.nodes;

import java.util.function.Consumer;
import javax.annotation.PostConstruct;

import de.iip_ecosphere.platform.services.environment.AbstractService;
import de.iip_ecosphere.platform.services.environment.MonitoringService;
import de.iip_ecosphere.platform.services.environment.spring.SpringAsyncServiceBase;
import de.iip_ecosphere.platform.services.environment.spring.Starter;
import de.iip_ecosphere.platform.services.environment.spring.metricsProvider.MetricsProvider;
import de.iip_ecosphere.platform.services.environment.switching.ServiceBase;
import de.iip_ecosphere.platform.support.logging.LoggerFactory;
import de.iip_ecosphere.platform.support.metrics.Counter;
import de.iip_ecosphere.platform.support.metrics.Timer;
import de.iip_ecosphere.platform.transport.Transport;

import iip.datatypes.RoutingTestData;
import iip.interfaces.RoutingSinkInterface;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.stream.function.StreamBridge;
import org.springframework.context.annotation.Bean;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.stereotype.Component;

/**
 * Spring Cloud Stream service frame for net node 'RoutingSink'.
 * Generated by: EASy-Producer.
 */
@Component
@ConditionalOnProperty(value = "iip.service.RoutingSink", havingValue = "true", matchIfMissing = true)
@EnableScheduling
public class RoutingSink extends SpringAsyncServiceBase {

    @Value("${iip.service.RoutingSink:true}")
    private String activated;
    @Autowired
    private StreamBridge streamBridge;
    private RoutingSinkInterface service;
    @Autowired
    private MetricsProvider metrics;
    private Counter serviceSent;
    private Counter serviceReceived;
    private Timer processingTime;
    private String appInstId = "";

    /**
     * Creates an instance.
     */
    public RoutingSink() {
        service = AbstractService.createInstance("de.iip_ecosphere.platform.test.apps.serviceImpl.routingTest.SinkImpl"
            , RoutingSinkInterface.class, "RoutingSink", "deployment.yml");
        appInstId = getAppInstIdSuffix(service, "_");
    }

    /**
     * Called when data arrived that shall be received (data sink).
     *
     * @return the data consumer functor
     */
    @Bean
    public Consumer<RoutingTestData> receiveRoutingTestData_RoutingSink() {
        return data -> {
            MetricsProvider.increaseCounterBy(serviceReceived, 1.0);
            processingTime.record(() -> service.processRoutingTestData(data));
        };
    }

    /**
     * Initializes the service when feasible in Spring lifecycle.
     */
    @PostConstruct
    public void initService() {
        if (null == activated || "".equals(activated) || "true".equals(activated)) {
            LoggerFactory.getLogger(getClass())
                .info("Initializing service RoutingSink: {}", activated);
            String iId;
            String sId;
            sId = Starter.getServiceId(service);
            iId = ServiceBase.getApplicationInstanceId(sId);
            if (iId == null || iId.length() == 0) {
                iId = "dflt";
            }
            serviceSent = metrics.createServiceSentCounter("RoutingSink", sId, "RoutingTestApp", iId);
            serviceReceived = metrics.createServiceReceivedCounter("RoutingSink", sId, "RoutingTestApp", iId);
            processingTime = metrics.createServiceProcessingTimer("RoutingSink", sId, "RoutingTestApp", iId);
            MonitoringService.setUp(service, metrics);
            Starter.mapService(service);
            service.attachRoutingCommandIngestor(data -> {
                MetricsProvider.increaseCounterBy(serviceSent, 1.0);
                Transport.send(c -> c.asyncSend("data_RoutingSink_RoutingCommand_RoutingTestApp" + appInstId, data), 
                    "RoutingSink", "processRoutingCommand_myRoutingConnector-in-0",
                    "processRoutingCommand_myRoutingSource-in-0","processRoutingCommand_ParallelRoutingProcessor1-in-"
                    + "0","processRoutingCommand_ParallelRoutingProcessor2-in-0",
                    "processRoutingCommand_RoutingProcessor-in-0");
            });
            createReceptionCallback("data_ParallelRoutingProcessor1_RoutingTestData_RoutingTestApp" + appInstId,
                receiveRoutingTestData_RoutingSink(), RoutingTestData.class, 
                "receiveRoutingTestData_RoutingSink-in-0");
            createReceptionCallback("data_ParallelRoutingProcessor2_RoutingTestData_RoutingTestApp" + appInstId,
                receiveRoutingTestData_RoutingSink(), RoutingTestData.class, 
                "receiveRoutingTestData_RoutingSink-in-0");
        }
    }

}
