/*
 * Decompiled with CFR 0.152.
 */
package jetbrains.exodus.log;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import jetbrains.exodus.ArrayByteIterable;
import jetbrains.exodus.ByteIterable;
import jetbrains.exodus.ByteIterator;
import jetbrains.exodus.ExodusException;
import jetbrains.exodus.InvalidSettingException;
import jetbrains.exodus.core.dataStructures.LongArrayList;
import jetbrains.exodus.core.dataStructures.hash.LongIterator;
import jetbrains.exodus.crypto.EnvKryptKt;
import jetbrains.exodus.crypto.InvalidCipherParametersException;
import jetbrains.exodus.crypto.StreamCipherProvider;
import jetbrains.exodus.io.Block;
import jetbrains.exodus.io.DataReader;
import jetbrains.exodus.io.DataWriter;
import jetbrains.exodus.io.RemoveBlockType;
import jetbrains.exodus.log.BlockNotFoundException;
import jetbrains.exodus.log.BufferedDataWriter;
import jetbrains.exodus.log.ByteIteratorWithAddress;
import jetbrains.exodus.log.CompressedUnsignedLongByteIterable;
import jetbrains.exodus.log.DataIterator;
import jetbrains.exodus.log.LogCache;
import jetbrains.exodus.log.LogConfig;
import jetbrains.exodus.log.LogFileSet;
import jetbrains.exodus.log.LogTestConfig;
import jetbrains.exodus.log.LogTip;
import jetbrains.exodus.log.LogUtil;
import jetbrains.exodus.log.Loggable;
import jetbrains.exodus.log.LoggableIterator;
import jetbrains.exodus.log.NewFileListener;
import jetbrains.exodus.log.NullLoggable;
import jetbrains.exodus.log.RandomAccessByteIterable;
import jetbrains.exodus.log.RandomAccessLoggable;
import jetbrains.exodus.log.RandomAccessLoggableAndArrayByteIterable;
import jetbrains.exodus.log.RandomAccessLoggableImpl;
import jetbrains.exodus.log.ReadBytesListener;
import jetbrains.exodus.log.RemoveFileListener;
import jetbrains.exodus.log.SeparateLogCache;
import jetbrains.exodus.log.SharedLogCache;
import jetbrains.exodus.log.TooBigLoggableException;
import jetbrains.exodus.util.DeferredIO;
import jetbrains.exodus.util.IdGenerator;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Log
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(Log.class);
    private static IdGenerator identityGenerator = new IdGenerator();
    private static volatile LogCache sharedCache = null;
    @NotNull
    private final LogConfig config;
    private final long created;
    @NotNull
    private final String location;
    final LogCache cache;
    private volatile boolean isClosing;
    private int logIdentity;
    @NotNull
    private final DataWriter baseWriter;
    private final AtomicReference<LogTip> tip;
    @Nullable
    private BufferedDataWriter bufferedWriter;
    private long lastSyncTicks;
    @NotNull
    private final DataReader reader;
    private final List<NewFileListener> newFileListeners;
    private final List<ReadBytesListener> readBytesListeners;
    private final List<RemoveFileListener> removeFileListeners;
    private final int cachePageSize;
    private final long fileLengthBound;
    @Nullable
    private LogTestConfig testConfig;

    public Log(@NotNull LogConfig config) {
        this.config = config;
        this.baseWriter = config.getWriter();
        this.tryLock();
        this.created = System.currentTimeMillis();
        this.cachePageSize = config.getCachePageSize();
        long fileLength = config.getFileSize() * 1024L;
        if (fileLength % (long)this.cachePageSize != 0L) {
            throw new InvalidSettingException("File size should be a multiple of cache page size.");
        }
        this.fileLengthBound = fileLength;
        LogFileSet.Mutable fileSetMutable = new LogFileSet.Immutable(fileLength).beginWrite();
        this.reader = config.getReader();
        this.reader.setLog(this);
        this.location = this.reader.getLocation();
        this.checkLogConsistency(fileSetMutable);
        LogFileSet.Immutable fileSetImmutable = fileSetMutable.endWrite();
        this.newFileListeners = new ArrayList<NewFileListener>(2);
        this.readBytesListeners = new ArrayList<ReadBytesListener>(2);
        this.removeFileListeners = new ArrayList<RemoveFileListener>(2);
        long memoryUsage = config.getMemoryUsage();
        boolean nonBlockingCache = config.isNonBlockingCache();
        if (memoryUsage != 0L) {
            this.cache = config.isSharedCache() ? Log.getSharedCache(memoryUsage, this.cachePageSize, nonBlockingCache) : new SeparateLogCache(memoryUsage, this.cachePageSize, nonBlockingCache);
        } else {
            int memoryUsagePercentage = config.getMemoryUsagePercentage();
            this.cache = config.isSharedCache() ? Log.getSharedCache(memoryUsagePercentage, this.cachePageSize, nonBlockingCache) : new SeparateLogCache(memoryUsagePercentage, this.cachePageSize, nonBlockingCache);
        }
        DeferredIO.getJobProcessor();
        this.isClosing = false;
        Long lastFileAddress = fileSetMutable.getMaximum();
        this.updateLogIdentity();
        if (lastFileAddress == null) {
            this.tip = new AtomicReference<LogTip>(new LogTip(this.fileLengthBound));
        } else {
            long currentHighAddress = lastFileAddress + this.reader.getBlock(lastFileAddress).length();
            long highPageAddress = this.getHighPageAddress(currentHighAddress);
            byte[] highPageContent = new byte[this.cachePageSize];
            LogTip tmpTip = new LogTip(highPageContent, highPageAddress, this.cachePageSize, currentHighAddress, currentHighAddress, fileSetImmutable);
            this.tip = new AtomicReference<LogTip>(tmpTip);
            int highPageSize = currentHighAddress == 0L ? 0 : this.readBytes(highPageContent, highPageAddress);
            LogTip proposedTip = new LogTip(highPageContent, highPageAddress, highPageSize, currentHighAddress, currentHighAddress, fileSetImmutable);
            this.tip.set(proposedTip);
            LoggableIterator lastFileLoggables = new LoggableIterator(this, lastFileAddress);
            long approvedHighAddress = lastFileAddress;
            try {
                while (lastFileLoggables.hasNext()) {
                    int dataLength;
                    RandomAccessLoggable loggable = (RandomAccessLoggable)lastFileLoggables.next();
                    int n = dataLength = NullLoggable.isNullLoggable(loggable) ? 0 : loggable.getDataLength();
                    if (dataLength > 0) {
                        ByteIteratorWithAddress data = loggable.getData().iterator();
                        for (int i = 0; i < dataLength; ++i) {
                            if (!data.hasNext()) {
                                throw new ExodusException("Can't read loggable fully" + LogUtil.getWrongAddressErrorMessage(data.getAddress(), this.fileLengthBound));
                            }
                            data.next();
                        }
                    }
                    approvedHighAddress = loggable.getAddress() + (long)loggable.length();
                }
            }
            catch (ExodusException e) {
                logger.error("Exception on Log recovery. Approved high address = " + approvedHighAddress, (Throwable)e);
            }
            if (approvedHighAddress < lastFileAddress || approvedHighAddress > currentHighAddress) {
                this.close();
                throw new InvalidCipherParametersException();
            }
            this.tip.set(proposedTip.withApprovedAddress(approvedHighAddress));
        }
        this.sync();
    }

    private void checkLogConsistency(LogFileSet.Mutable fileSetMutable) {
        Block[] blocks = this.reader.getBlocks();
        for (int i = 0; i < blocks.length; ++i) {
            Block block = blocks[i];
            long address = block.getAddress();
            long blockLength = block.length();
            String clearLogReason = null;
            if (blockLength > this.fileLengthBound || i < blocks.length - 1 && blockLength != this.fileLengthBound) {
                clearLogReason = "Unexpected file length" + LogUtil.getWrongAddressErrorMessage(address, this.fileLengthBound);
            }
            if (clearLogReason == null && address != this.getFileAddress(address)) {
                if (!this.config.isClearInvalidLog()) {
                    throw new ExodusException("Unexpected file address " + LogUtil.getLogFilename(address) + LogUtil.getWrongAddressErrorMessage(address, this.fileLengthBound));
                }
                clearLogReason = "Unexpected file address " + LogUtil.getLogFilename(address) + LogUtil.getWrongAddressErrorMessage(address, this.fileLengthBound);
            }
            if (clearLogReason != null) {
                if (!this.config.isClearInvalidLog()) {
                    throw new ExodusException(clearLogReason);
                }
                logger.error("Clearing log due to: " + clearLogReason);
                fileSetMutable.clear();
                this.reader.clear();
                break;
            }
            fileSetMutable.add(address);
        }
    }

    @NotNull
    public LogConfig getConfig() {
        return this.config;
    }

    public long getCreated() {
        return this.created;
    }

    @NotNull
    public String getLocation() {
        return this.location;
    }

    public long getFileLengthBound() {
        return this.fileLengthBound;
    }

    public long getNumberOfFiles() {
        return this.getTip().logFileSet.size();
    }

    public long[] getAllFileAddresses() {
        return this.getTip().logFileSet.getArray();
    }

    public long getHighAddress() {
        return this.getTip().highAddress;
    }

    public LogTip setHighAddress(LogTip logTip, long highAddress) {
        return this.setHighAddress(logTip, highAddress, null);
    }

    public LogTip setHighAddress(LogTip logTip, long highAddress, LogFileSet fileSet) {
        LogTip updatedTip;
        if (highAddress > logTip.highAddress) {
            throw new ExodusException("Only can decrease high address");
        }
        if (highAddress == logTip.highAddress) {
            if (this.bufferedWriter != null) {
                throw new IllegalStateException("Unexpected write in progress");
            }
            return logTip;
        }
        LogFileSet.Mutable fileSetMutable = logTip.logFileSet.beginWrite();
        LogTestConfig testConfig = this.testConfig;
        if (testConfig != null && testConfig.isSettingHighAddressDenied()) {
            throw new ExodusException("Setting high address is denied");
        }
        this.closeWriter();
        LongArrayList blocksToDelete = new LongArrayList();
        long blockToTruncate = -1L;
        for (long blockAddress : (fileSet == null ? fileSetMutable : fileSet).getArray()) {
            if (blockAddress <= highAddress) {
                blockToTruncate = blockAddress;
                break;
            }
            blocksToDelete.add(blockAddress);
        }
        for (int i = 0; i < blocksToDelete.size(); ++i) {
            this.removeFile(blocksToDelete.get(i), RemoveBlockType.Delete, fileSetMutable);
        }
        if (blockToTruncate >= 0L) {
            this.truncateFile(blockToTruncate, highAddress - blockToTruncate);
        }
        if (fileSetMutable.isEmpty()) {
            this.updateLogIdentity();
            updatedTip = new LogTip(this.fileLengthBound);
        } else {
            long oldHighPageAddress = logTip.pageAddress;
            long approvedHighAddress = logTip.approvedHighAddress;
            if (highAddress < approvedHighAddress) {
                approvedHighAddress = highAddress;
            }
            long highPageAddress = this.getHighPageAddress(highAddress);
            LogFileSet.Immutable fileSetImmutable = fileSetMutable.endWrite();
            int highPageSize = (int)(highAddress - highPageAddress);
            if (oldHighPageAddress == highPageAddress) {
                updatedTip = logTip.withResize(highPageSize, highAddress, approvedHighAddress, fileSetImmutable);
            } else {
                this.updateLogIdentity();
                byte[] highPageContent = new byte[this.cachePageSize];
                if (highPageSize > 0 && this.readBytes(highPageContent, highPageAddress) < highPageSize) {
                    throw new ExodusException("Can't read expected high page bytes");
                }
                updatedTip = new LogTip(highPageContent, highPageAddress, highPageSize, highAddress, approvedHighAddress, fileSetImmutable);
            }
        }
        this.compareAndSetTip(logTip, updatedTip);
        this.bufferedWriter = null;
        return updatedTip;
    }

    private void closeWriter() {
        if (this.bufferedWriter != null) {
            throw new IllegalStateException("Unexpected write in progress");
        }
        this.baseWriter.close();
    }

    public LogTip getTip() {
        return this.tip.get();
    }

    public LogTip beginWrite() {
        BufferedDataWriter writer;
        this.bufferedWriter = writer = new BufferedDataWriter(this, this.baseWriter, this.reader, this.getTip());
        return writer.getStartingTip();
    }

    public void abortWrite() {
        this.bufferedWriter = null;
    }

    public void revertWrite(LogTip logTip) {
        LogFileSet.Mutable fileSet = this.ensureWriter().getFileSetMutable();
        this.abortWrite();
        this.setHighAddress(logTip, logTip.highAddress, fileSet);
    }

    public long getWrittenHighAddress() {
        return this.ensureWriter().getHighAddress();
    }

    public LogTip endWrite() {
        BufferedDataWriter writer = this.ensureWriter();
        LogTip logTip = writer.getStartingTip();
        LogTip updatedTip = writer.getUpdatedTip();
        this.compareAndSetTip(logTip, updatedTip);
        this.bufferedWriter = null;
        return updatedTip;
    }

    public LogTip compareAndSetTip(LogTip logTip, LogTip updatedTip) {
        if (!this.tip.compareAndSet(logTip, updatedTip)) {
            throw new ExodusException("write start/finish mismatch");
        }
        return updatedTip;
    }

    public long getLowAddress() {
        Long result = this.getTip().logFileSet.getMinimum();
        return result == null ? -1L : result;
    }

    public long getFileAddress(long address) {
        return address - address % this.fileLengthBound;
    }

    public long getHighFileAddress() {
        return this.getFileAddress(this.getHighAddress());
    }

    public long getNextFileAddress(long fileAddress) {
        LongIterator files = this.getTip().logFileSet.getFilesFrom(fileAddress);
        if (files.hasNext()) {
            long result = files.nextLong();
            if (result != fileAddress) {
                throw new ExodusException("There is no file by address " + fileAddress);
            }
            if (files.hasNext()) {
                return files.nextLong();
            }
        }
        return -1L;
    }

    public boolean isLastFileAddress(long address, LogTip logTip) {
        return this.getFileAddress(address) == this.getFileAddress(logTip.highAddress);
    }

    public boolean isLastWrittenFileAddress(long address) {
        return this.getFileAddress(address) == this.getFileAddress(this.getWrittenHighAddress());
    }

    public boolean hasAddress(long address) {
        long fileAddress = this.getFileAddress(address);
        LogTip logTip = this.getTip();
        LongIterator files = logTip.logFileSet.getFilesFrom(fileAddress);
        if (!files.hasNext()) {
            return false;
        }
        long leftBound = files.nextLong();
        return leftBound == fileAddress && leftBound + this.getFileSize(leftBound, logTip) > address;
    }

    public boolean hasAddressRange(long from, long to) {
        long fileAddress = this.getFileAddress(from);
        LogTip logTip = this.getTip();
        LongIterator files = logTip.logFileSet.getFilesFrom(fileAddress);
        do {
            if (files.hasNext() && files.nextLong() == fileAddress) continue;
            return false;
        } while ((fileAddress += this.getFileSize(fileAddress, logTip)) > from && fileAddress <= to);
        return true;
    }

    public long getFileSize(long fileAddress) {
        return this.getFileSize(fileAddress, this.getTip());
    }

    public long getFileSize(long fileAddress, LogTip logTip) {
        if (!this.isLastFileAddress(fileAddress, logTip)) {
            return this.fileLengthBound;
        }
        return this.getLastFileSize(fileAddress, logTip);
    }

    private long getLastFileSize(long fileAddress, LogTip logTip) {
        long highAddress = logTip.highAddress;
        long result = highAddress % this.fileLengthBound;
        if (result == 0L && highAddress != fileAddress) {
            return this.fileLengthBound;
        }
        return result;
    }

    public long getDiskUsage() {
        LogTip tip = this.getTip();
        long[] allFiles = tip.getAllFiles();
        int filesCount = allFiles.length;
        return filesCount == 0 ? 0L : (long)(filesCount - 1) * this.fileLengthBound + this.getLastFileSize(allFiles[filesCount - 1], tip);
    }

    byte[] getHighPage(long alignedAddress) {
        LogTip tip = this.getTip();
        if (tip.pageAddress == alignedAddress && tip.count >= 0) {
            return tip.bytes;
        }
        return null;
    }

    public byte[] getCachedPage(long pageAddress) {
        return this.cache.getPage(this, pageAddress);
    }

    public final int getCachePageSize() {
        return this.cachePageSize;
    }

    public float getCacheHitRate() {
        return this.cache == null ? 0.0f : this.cache.hitRate();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addNewFileListener(@NotNull NewFileListener listener) {
        List<NewFileListener> list = this.newFileListeners;
        synchronized (list) {
            this.newFileListeners.add(listener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addReadBytesListener(@NotNull ReadBytesListener listener) {
        List<ReadBytesListener> list = this.readBytesListeners;
        synchronized (list) {
            this.readBytesListeners.add(listener);
        }
    }

    @NotNull
    public RandomAccessLoggable read(long address) {
        return this.read(this.readIteratorFrom(address), address);
    }

    @NotNull
    public RandomAccessLoggable read(DataIterator it) {
        return this.read(it, it.getHighAddress());
    }

    public byte getWrittenLoggableType(long address, byte max) {
        return this.ensureWriter().getByte(address, max);
    }

    @NotNull
    public RandomAccessLoggable read(DataIterator it, long address) {
        byte type = (byte)(it.next() ^ 0x80);
        if (NullLoggable.isNullLoggable(type)) {
            return new NullLoggable(address);
        }
        return this.read(type, it, address);
    }

    @NotNull
    public RandomAccessLoggable readNotNull(DataIterator it, long address) {
        return this.read((byte)(it.next() ^ 0x80), it, address);
    }

    @NotNull
    private RandomAccessLoggable read(byte type, DataIterator it, long address) {
        int structureId = CompressedUnsignedLongByteIterable.getInt(it);
        int dataLength = CompressedUnsignedLongByteIterable.getInt(it);
        long dataAddress = it.getHighAddress();
        if (dataLength > 0 && it.availableInCurrentPage(dataLength)) {
            return new RandomAccessLoggableAndArrayByteIterable(address, type, structureId, dataAddress, it.getCurrentPage(), it.getOffset(), dataLength);
        }
        RandomAccessByteIterable data = new RandomAccessByteIterable(dataAddress, this);
        return new RandomAccessLoggableImpl(address, type, data, dataLength, structureId);
    }

    public LoggableIterator getLoggableIterator(long startAddress) {
        return new LoggableIterator(this, startAddress);
    }

    public long tryWrite(Loggable loggable) {
        return this.tryWrite(loggable.getType(), loggable.getStructureId(), loggable.getData());
    }

    public long tryWrite(byte type, int structureId, ByteIterable data) {
        long result = this.writeContinuously(type, structureId, data);
        if (result < 0L) {
            this.doPadWithNulls();
        }
        return result;
    }

    public long write(Loggable loggable) {
        return this.write(loggable.getType(), loggable.getStructureId(), loggable.getData());
    }

    public long write(byte type, int structureId, ByteIterable data) {
        long result = this.writeContinuously(type, structureId, data);
        if (result < 0L) {
            this.doPadWithNulls();
            result = this.writeContinuously(type, structureId, data);
            if (result < 0L) {
                throw new TooBigLoggableException();
            }
        }
        return result;
    }

    @Nullable
    public Loggable getFirstLoggableOfType(int type) {
        LogTip logTip = this.getTip();
        LongIterator files = logTip.logFileSet.getFilesFrom(0L);
        long approvedHighAddress = logTip.approvedHighAddress;
        block0: while (files.hasNext()) {
            Loggable loggable;
            long fileAddress = files.nextLong();
            LoggableIterator it = this.getLoggableIterator(fileAddress);
            while (it.hasNext() && (loggable = (Loggable)it.next()) != null && loggable.getAddress() < fileAddress + this.fileLengthBound) {
                if (loggable.getType() == type) {
                    return loggable;
                }
                if (loggable.getAddress() + (long)loggable.length() != approvedHighAddress) continue;
                continue block0;
            }
        }
        return null;
    }

    @Nullable
    public Loggable getLastLoggableOfType(int type) {
        Loggable result = null;
        LogTip logTip = this.getTip();
        long approvedHighAddress = logTip.approvedHighAddress;
        block0: for (long fileAddress : logTip.logFileSet.getArray()) {
            Loggable loggable;
            if (result != null) break;
            LoggableIterator it = this.getLoggableIterator(fileAddress);
            while (it.hasNext() && (loggable = (Loggable)it.next()) != null && loggable.getAddress() < fileAddress + this.fileLengthBound) {
                if (loggable.getType() == type) {
                    result = loggable;
                }
                if (loggable.getAddress() + (long)loggable.length() != approvedHighAddress) continue;
                continue block0;
            }
        }
        return result;
    }

    public Loggable getLastLoggableOfTypeBefore(int type, long beforeAddress, LogTip logTip) {
        Loggable result = null;
        for (long fileAddress : logTip.logFileSet.getArray()) {
            long address;
            Loggable loggable;
            if (result != null) break;
            if (fileAddress >= beforeAddress) continue;
            LoggableIterator it = this.getLoggableIterator(fileAddress);
            while (it.hasNext() && (loggable = (Loggable)it.next()) != null && (address = loggable.getAddress()) < beforeAddress && address < fileAddress + this.fileLengthBound) {
                if (loggable.getType() != type) continue;
                result = loggable;
            }
        }
        return result;
    }

    public boolean isImmutableFile(long fileAddress) {
        return fileAddress + this.fileLengthBound <= this.getTip().approvedHighAddress;
    }

    public void flush() {
        this.flush(false);
    }

    public void flush(boolean forceSync) {
        BufferedDataWriter writer = this.ensureWriter();
        writer.flush();
        if (forceSync || this.config.isDurableWrite()) {
            this.sync();
        }
    }

    public void sync() {
        if (!this.config.isFsyncSuppressed()) {
            this.baseWriter.sync();
            this.lastSyncTicks = System.currentTimeMillis();
        }
    }

    @Override
    public void close() {
        LogTip logTip = this.getTip();
        this.isClosing = true;
        this.sync();
        this.reader.close();
        this.closeWriter();
        this.compareAndSetTip(logTip, new LogTip(this.fileLengthBound, logTip.pageAddress, logTip.highAddress));
        this.release();
    }

    public boolean isClosing() {
        return this.isClosing;
    }

    public void release() {
        this.baseWriter.release();
    }

    public LogTip clear() {
        LogTip logTip = this.getTip();
        this.closeWriter();
        this.cache.clear();
        this.reader.clear();
        LogTip updatedTip = new LogTip(this.fileLengthBound);
        this.compareAndSetTip(logTip, updatedTip);
        this.bufferedWriter = null;
        this.updateLogIdentity();
        return updatedTip;
    }

    public void forgetFile(long address) {
        this.beginWrite();
        this.forgetFiles(new long[]{address});
        this.endWrite();
    }

    public void forgetFiles(long[] files) {
        LogFileSet.Mutable fileSetMutable = this.ensureWriter().getFileSetMutable();
        for (long file : files) {
            fileSetMutable.remove(file);
        }
    }

    public void removeFile(long address) {
        this.removeFile(address, RemoveBlockType.Delete, null);
    }

    public void removeFile(long address, @NotNull RemoveBlockType rbt) {
        this.removeFile(address, rbt, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeFile(long address, @NotNull RemoveBlockType rbt, @Nullable LogFileSet.Mutable logFileSetMutable) {
        RemoveFileListener[] removeFileListenerArray = this.removeFileListeners;
        synchronized (this.removeFileListeners) {
            RemoveFileListener[] listeners = this.removeFileListeners.toArray(new RemoveFileListener[this.removeFileListeners.size()]);
            // ** MonitorExit[var6_4] (shouldn't be in output)
            for (RemoveFileListener listener : listeners) {
                listener.beforeRemoveFile(address);
            }
            try {
                this.reader.removeBlock(address, rbt);
                if (logFileSetMutable != null) {
                    logFileSetMutable.remove(address);
                }
                for (long offset = 0L; offset < this.fileLengthBound; offset += (long)this.cachePageSize) {
                    this.cache.removePage(this, address + offset);
                }
            }
            finally {
                for (RemoveFileListener listener : listeners) {
                    listener.afterRemoveFile(address);
                }
            }
            return;
        }
    }

    @NotNull
    public BufferedDataWriter ensureWriter() {
        BufferedDataWriter writer = this.bufferedWriter;
        if (writer == null) {
            throw new ExodusException("write not in progress");
        }
        return writer;
    }

    private void truncateFile(long address, long length) {
        this.reader.truncateBlock(address, length);
        this.baseWriter.openOrCreateBlock(address, length);
        for (long offset = length - length % (long)this.cachePageSize; offset < this.fileLengthBound; offset += (long)this.cachePageSize) {
            this.cache.removePage(this, address + offset);
        }
    }

    public void padWithNulls() {
        this.beforeWrite(this.ensureWriter());
        this.doPadWithNulls();
    }

    void doPadWithNulls() {
        byte[] cachedTailPage;
        BufferedDataWriter writer = this.ensureWriter();
        long bytesToWrite = this.fileLengthBound - writer.getLastWrittenFileLength(this.fileLengthBound);
        if (bytesToWrite == 0L) {
            throw new ExodusException("Nothing to pad");
        }
        if (bytesToWrite >= (long)this.cachePageSize && (cachedTailPage = LogCache.getCachedTailPage(this.cachePageSize)) != null) {
            do {
                writer.write(cachedTailPage, this.cachePageSize);
                writer.incHighAddress(this.cachePageSize);
            } while ((bytesToWrite -= (long)this.cachePageSize) >= (long)this.cachePageSize);
        }
        if (bytesToWrite == 0L) {
            writer.commit();
            this.closeFullFileFileIfNecessary(writer);
        } else {
            while (bytesToWrite-- > 0L) {
                this.writeContinuously(NullLoggable.create());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void invalidateSharedCache() {
        Class<Log> clazz = Log.class;
        synchronized (Log.class) {
            sharedCache = null;
            // ** MonitorExit[var0] (shouldn't be in output)
            return;
        }
    }

    public int getIdentity() {
        return this.logIdentity;
    }

    int readBytes(byte[] output, long address) {
        long fileAddress = this.getFileAddress(address);
        LogTip logTip = this.getTip();
        LongIterator files = logTip.logFileSet.getFilesFrom(fileAddress);
        if (files.hasNext()) {
            long leftBound = files.nextLong();
            long fileSize = this.getFileSize(leftBound, logTip);
            if (leftBound == fileAddress && fileAddress + fileSize > address) {
                Block block = this.reader.getBlock(fileAddress);
                int readBytes = block.read(output, address - fileAddress, output.length);
                StreamCipherProvider cipherProvider = this.config.getCipherProvider();
                if (cipherProvider != null) {
                    EnvKryptKt.cryptBlocksMutable(cipherProvider, this.config.getCipherKey(), this.config.getCipherBasicIV(), address, output, 0, readBytes, 1024);
                }
                this.notifyReadBytes(output, readBytes);
                return readBytes;
            }
            if (fileAddress < logTip.logFileSet.getMinimum()) {
                BlockNotFoundException.raise("Address is out of log space, underflow", this, address);
            }
            if (fileAddress >= logTip.logFileSet.getMaximum()) {
                BlockNotFoundException.raise("Address is out of log space, overflow", this, address);
            }
        }
        BlockNotFoundException.raise(this, address);
        return 0;
    }

    DataIterator readIteratorFrom(long address) {
        return new DataIterator(this, address);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    @NotNull
    private static LogCache getSharedCache(long memoryUsage, int pageSize, boolean nonBlocking) {
        LogCache result = sharedCache;
        if (result == null) {
            Class<Log> clazz = Log.class;
            // MONITORENTER : jetbrains.exodus.log.Log.class
            if (sharedCache == null) {
                sharedCache = new SharedLogCache(memoryUsage, pageSize, nonBlocking);
            }
            result = sharedCache;
            // MONITOREXIT : clazz
        }
        Log.checkCachePageSize(pageSize, result);
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    @NotNull
    private static LogCache getSharedCache(int memoryUsagePercentage, int pageSize, boolean nonBlocking) {
        LogCache result = sharedCache;
        if (result == null) {
            Class<Log> clazz = Log.class;
            // MONITORENTER : jetbrains.exodus.log.Log.class
            if (sharedCache == null) {
                sharedCache = new SharedLogCache(memoryUsagePercentage, pageSize, nonBlocking);
            }
            result = sharedCache;
            // MONITOREXIT : clazz
        }
        Log.checkCachePageSize(pageSize, result);
        return result;
    }

    private static void checkCachePageSize(int pageSize, @NotNull LogCache result) {
        if (result.pageSize != pageSize) {
            throw new ExodusException("SharedLogCache was created with page size " + result.pageSize + " and then requested with page size " + pageSize + ". EnvironmentConfig.LOG_CACHE_PAGE_SIZE was set manually.");
        }
    }

    private void tryLock() {
        long lockTimeout = this.config.getLockTimeout();
        if (!this.baseWriter.lock(lockTimeout)) {
            throw new ExodusException("Can't acquire environment lock after " + lockTimeout + " ms.\n\n Lock owner info: \n" + this.baseWriter.lockInfo());
        }
    }

    public long getHighPageAddress(long highAddress) {
        int alignment = (int)highAddress & this.cachePageSize - 1;
        if (alignment == 0 && highAddress > 0L) {
            alignment = this.cachePageSize;
        }
        return highAddress - (long)alignment;
    }

    public LongIterator getFilesFrom(LogTip logTip, Long fileAddress) {
        return logTip.logFileSet.getFilesFrom(fileAddress);
    }

    public long writeContinuously(Loggable loggable) {
        return this.writeContinuously(loggable.getType(), loggable.getStructureId(), loggable.getData());
    }

    public long writeContinuously(byte type, int structureId, ByteIterable data) {
        BufferedDataWriter writer = this.ensureWriter();
        long result = this.beforeWrite(writer);
        boolean isNull = NullLoggable.isNullLoggable(type);
        int recordLength = 1;
        if (isNull) {
            writer.write((byte)(type ^ 0x80));
        } else {
            ByteIterable structureIdIterable = CompressedUnsignedLongByteIterable.getIterable(structureId);
            int dataLength = data.getLength();
            ByteIterable dataLengthIterable = CompressedUnsignedLongByteIterable.getIterable(dataLength);
            recordLength += structureIdIterable.getLength();
            recordLength += dataLengthIterable.getLength();
            if ((long)(recordLength += dataLength) > this.fileLengthBound - writer.getLastWrittenFileLength(this.fileLengthBound)) {
                return -1L;
            }
            writer.write((byte)(type ^ 0x80));
            Log.writeByteIterable(writer, structureIdIterable);
            Log.writeByteIterable(writer, dataLengthIterable);
            if (dataLength > 0) {
                Log.writeByteIterable(writer, data);
            }
        }
        writer.commit();
        writer.incHighAddress(recordLength);
        this.closeFullFileFileIfNecessary(writer);
        return result;
    }

    public long writeContinuously(byte[] data, int count) {
        BufferedDataWriter writer = this.ensureWriter();
        long result = this.beforeWrite(writer);
        if ((long)count > this.fileLengthBound - writer.getLastWrittenFileLength(this.fileLengthBound)) {
            return -1L;
        }
        writer.write(data, count);
        writer.commit();
        writer.incHighAddress(count);
        this.closeFullFileFileIfNecessary(writer);
        return result;
    }

    private long beforeWrite(BufferedDataWriter writer) {
        long maxHighAddress;
        long result = writer.getHighAddress();
        LogTestConfig testConfig = this.testConfig;
        if (testConfig != null && (maxHighAddress = testConfig.getMaxHighAddress()) >= 0L && result >= maxHighAddress) {
            throw new ExodusException("Can't write more than " + maxHighAddress);
        }
        if (!this.baseWriter.isOpen()) {
            boolean fileCreated;
            long fileAddress = this.getFileAddress(result);
            writer.openOrCreateBlock(fileAddress, writer.getLastWrittenFileLength(this.fileLengthBound));
            boolean bl = fileCreated = !writer.getFileSetMutable().contains(fileAddress);
            if (fileCreated) {
                writer.getFileSetMutable().add(fileAddress);
            }
            if (fileCreated) {
                this.baseWriter.syncDirectory();
                this.notifyFileCreated(fileAddress);
            }
        }
        return result;
    }

    private void closeFullFileFileIfNecessary(BufferedDataWriter writer) {
        boolean shouldCreateNewFile;
        boolean bl = shouldCreateNewFile = writer.getLastWrittenFileLength(this.fileLengthBound) == 0L;
        if (shouldCreateNewFile) {
            Long lastFile;
            this.flush(true);
            this.baseWriter.close();
            if (this.config.isFullFileReadonly() && (lastFile = writer.getFileSetMutable().getMaximum()) != null) {
                Block block = this.reader.getBlock(lastFile);
                if (block.length() < this.fileLengthBound) {
                    throw new IllegalStateException("file too short");
                }
                block.setReadOnly();
            }
        } else if (System.currentTimeMillis() > this.lastSyncTicks + this.config.getSyncPeriod()) {
            this.flush(true);
        }
    }

    public void setLogTestConfig(@Nullable LogTestConfig testConfig) {
        this.testConfig = testConfig;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void notifyFileCreated(long fileAddress) {
        NewFileListener[] newFileListenerArray = this.newFileListeners;
        synchronized (this.newFileListeners) {
            NewFileListener[] listeners = this.newFileListeners.toArray(new NewFileListener[this.newFileListeners.size()]);
            // ** MonitorExit[var4_2] (shouldn't be in output)
            for (NewFileListener listener : listeners) {
                listener.fileCreated(fileAddress);
            }
            return;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void notifyReadBytes(byte[] bytes, int count) {
        ReadBytesListener[] readBytesListenerArray = this.readBytesListeners;
        synchronized (this.readBytesListeners) {
            ReadBytesListener[] listeners = this.readBytesListeners.toArray(new ReadBytesListener[this.readBytesListeners.size()]);
            // ** MonitorExit[var4_3] (shouldn't be in output)
            for (ReadBytesListener listener : listeners) {
                listener.bytesRead(bytes, count);
            }
            return;
        }
    }

    private static void writeByteIterable(BufferedDataWriter writer, ByteIterable iterable) {
        int length = iterable.getLength();
        if (iterable instanceof ArrayByteIterable) {
            byte[] bytes = iterable.getBytesUnsafe();
            if (length == 1) {
                writer.write(bytes[0]);
            } else {
                writer.write(bytes, length);
            }
        } else if (length >= 3) {
            writer.write(iterable.getBytesUnsafe(), length);
        } else {
            ByteIterator iterator = iterable.iterator();
            writer.write(iterator.next());
            if (length == 2) {
                writer.write(iterator.next());
            }
        }
    }

    private void updateLogIdentity() {
        this.logIdentity = identityGenerator.nextId();
    }
}

