/*
 * Decompiled with CFR 0.152.
 */
package com.hivemq.persistence.local.memory;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.hivemq.annotations.ExecuteInSingleWriter;
import com.hivemq.bootstrap.ioc.lazysingleton.LazySingleton;
import com.hivemq.configuration.service.InternalConfigurations;
import com.hivemq.extension.sdk.api.annotations.NotNull;
import com.hivemq.extension.sdk.api.annotations.Nullable;
import com.hivemq.extensions.iteration.BucketChunkResult;
import com.hivemq.logging.EventLog;
import com.hivemq.metrics.HiveMQMetrics;
import com.hivemq.metrics.MetricsHolder;
import com.hivemq.persistence.NoSessionException;
import com.hivemq.persistence.PersistenceEntry;
import com.hivemq.persistence.clientsession.ClientSession;
import com.hivemq.persistence.clientsession.ClientSessionWill;
import com.hivemq.persistence.clientsession.PendingWillMessages;
import com.hivemq.persistence.exception.InvalidSessionExpiryIntervalException;
import com.hivemq.persistence.local.ClientSessionLocalPersistence;
import com.hivemq.persistence.local.xodus.bucket.BucketUtils;
import com.hivemq.persistence.payload.PublishPayloadPersistence;
import com.hivemq.util.ObjectMemoryEstimation;
import com.hivemq.util.ThreadPreConditions;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@LazySingleton
public class ClientSessionMemoryLocalPersistence
implements ClientSessionLocalPersistence {
    @NotNull
    private static final Logger log = LoggerFactory.getLogger(ClientSessionMemoryLocalPersistence.class);
    @NotNull
    private final PublishPayloadPersistence payloadPersistence;
    @NotNull
    private final MetricsHolder metricsHolder;
    @NotNull
    private final EventLog eventLog;
    @NotNull
    private final @NotNull Map<String, PersistenceEntry<ClientSession>> @NotNull [] buckets;
    @NotNull
    private final AtomicInteger sessionsCount = new AtomicInteger(0);
    @NotNull
    private final AtomicLong currentMemorySize = new AtomicLong();
    private final int bucketCount;

    @Inject
    ClientSessionMemoryLocalPersistence(@NotNull PublishPayloadPersistence payloadPersistence, @NotNull MetricRegistry metricRegistry, @NotNull MetricsHolder metricsHolder, @NotNull EventLog eventLog) {
        this.payloadPersistence = payloadPersistence;
        this.metricsHolder = metricsHolder;
        this.eventLog = eventLog;
        this.bucketCount = InternalConfigurations.PERSISTENCE_BUCKET_COUNT.get();
        this.buckets = new Map[this.bucketCount];
        for (int i = 0; i < this.bucketCount; ++i) {
            this.buckets[i] = new HashMap<String, PersistenceEntry<ClientSession>>();
        }
        metricRegistry.register(HiveMQMetrics.CLIENT_SESSIONS_MEMORY_PERSISTENCE_TOTAL_SIZE.name(), (Metric)((Gauge)this.currentMemorySize::get));
    }

    @NotNull
    private Map<String, PersistenceEntry<ClientSession>> getBucket(int bucketIndex) {
        Preconditions.checkArgument((bucketIndex <= this.bucketCount ? 1 : 0) != 0, (Object)"Bucket must be less or equal than bucketCount");
        return this.buckets[bucketIndex];
    }

    @Override
    @Nullable
    public ClientSession getSession(@NotNull String clientId, int bucketIndex, boolean checkExpired) {
        return this.getSession(clientId, bucketIndex, checkExpired, true);
    }

    @Override
    @Nullable
    public ClientSession getSession(@NotNull String clientId, int bucketIndex) {
        return this.getSession(clientId, bucketIndex, true, true);
    }

    @Override
    @Nullable
    public ClientSession getSession(@NotNull String clientId, boolean checkExpired) {
        int bucketIndex = BucketUtils.getBucket(clientId, this.bucketCount);
        return this.getSession(clientId, bucketIndex, checkExpired, true);
    }

    @Override
    @Nullable
    public ClientSession getSession(@NotNull String clientId, boolean checkExpired, boolean includeWill) {
        int bucketIndex = BucketUtils.getBucket(clientId, this.bucketCount);
        return this.getSession(clientId, bucketIndex, checkExpired, includeWill);
    }

    @Override
    @Nullable
    public ClientSession getSession(@NotNull String clientId) {
        int bucketIndex = BucketUtils.getBucket(clientId, this.bucketCount);
        return this.getSession(clientId, bucketIndex, true, true);
    }

    @Nullable
    private ClientSession getSession(@NotNull String clientId, int bucketIndex, boolean checkExpired, boolean includeWill) {
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        PersistenceEntry<ClientSession> storedSession = bucket.get(clientId);
        if (storedSession == null) {
            return null;
        }
        ClientSession clientSession = storedSession.getObject().deepCopy();
        if (checkExpired && clientSession.isExpired(System.currentTimeMillis() - storedSession.getTimestamp())) {
            return null;
        }
        if (includeWill) {
            this.loadWillPayload(clientSession);
        }
        return clientSession;
    }

    @Override
    @Nullable
    public Long getTimestamp(@NotNull String clientId) {
        int bucketIndex = BucketUtils.getBucket(clientId, this.bucketCount);
        return this.getTimestamp(clientId, bucketIndex);
    }

    @Override
    @Nullable
    public Long getTimestamp(@NotNull String clientId, int bucketIndex) {
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        PersistenceEntry<ClientSession> storedSession = bucket.get(clientId);
        if (storedSession == null) {
            return null;
        }
        return storedSession.getTimestamp();
    }

    @Override
    @ExecuteInSingleWriter
    public void put(@NotNull String clientId, @NotNull ClientSession newClientSession, long timestamp, int bucketIndex) {
        Preconditions.checkNotNull((Object)clientId, (Object)"Client id must not be null");
        Preconditions.checkNotNull((Object)newClientSession, (Object)"Client session must not be null");
        Preconditions.checkArgument((timestamp > 0L ? 1 : 0) != 0, (Object)"Timestamp must be greater than 0");
        ThreadPreConditions.startsWith("single-writer");
        Map<String, PersistenceEntry<ClientSession>> sessions = this.getBucket(bucketIndex);
        ClientSession usedSession = newClientSession.deepCopy();
        sessions.compute(clientId, (ignored, storedSession) -> {
            boolean addClientIdSize;
            if (storedSession == null) {
                this.sessionsCount.incrementAndGet();
                addClientIdSize = true;
            } else {
                ClientSession oldSession = (ClientSession)storedSession.getObject();
                this.currentMemorySize.addAndGet(-storedSession.getEstimatedSize());
                this.removeWillReference(oldSession);
                boolean oldSessionIsPersistent = ClientSessionMemoryLocalPersistence.isPersistent(oldSession);
                if (!oldSessionIsPersistent && !oldSession.isConnected()) {
                    this.sessionsCount.incrementAndGet();
                }
                addClientIdSize = false;
            }
            ClientSessionWill newWill = newClientSession.getWillPublish();
            if (newWill != null) {
                this.metricsHolder.getStoredWillMessagesCount().inc();
                this.payloadPersistence.add(newWill.getPayload(), newWill.getPublishId());
            }
            PersistenceEntry<ClientSession> newEntry = new PersistenceEntry<ClientSession>(usedSession, timestamp);
            this.currentMemorySize.addAndGet(newEntry.getEstimatedSize());
            if (addClientIdSize) {
                this.currentMemorySize.addAndGet(ObjectMemoryEstimation.stringSize(clientId));
            }
            return newEntry;
        });
    }

    @Override
    @ExecuteInSingleWriter
    @NotNull
    public ClientSession disconnect(@NotNull String clientId, long timestamp, boolean sendWill, int bucketIndex, long sessionExpiryInterval) {
        ThreadPreConditions.startsWith("single-writer");
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        ClientSession storedSession = ((ClientSession)bucket.compute(clientId, (ignored, oldEntry) -> {
            ClientSession newSession;
            if (oldEntry == null) {
                ClientSession clientSession = new ClientSession(false, 0L);
                PersistenceEntry<ClientSession> persistenceEntry = new PersistenceEntry<ClientSession>(clientSession, timestamp);
                this.currentMemorySize.addAndGet(persistenceEntry.getEstimatedSize() + ObjectMemoryEstimation.stringSize(clientId));
                return persistenceEntry;
            }
            this.currentMemorySize.addAndGet(-oldEntry.getEstimatedSize());
            ClientSession oldSession = (ClientSession)oldEntry.getObject();
            if (sendWill) {
                newSession = oldSession;
            } else {
                this.removeWillReference(oldSession);
                newSession = oldSession.copyWithoutWill();
            }
            if (sessionExpiryInterval != Long.MAX_VALUE) {
                newSession.setSessionExpiryIntervalSec(sessionExpiryInterval);
            }
            if (newSession.isConnected() && !ClientSessionMemoryLocalPersistence.isPersistent(newSession)) {
                this.sessionsCount.decrementAndGet();
            }
            newSession.setConnected(false);
            this.loadWillPayload(newSession, false);
            PersistenceEntry<ClientSession> newEntry = new PersistenceEntry<ClientSession>(newSession, timestamp);
            this.currentMemorySize.addAndGet(newEntry.getEstimatedSize());
            return newEntry;
        }).getObject()).deepCopy();
        this.loadWillPayload(storedSession);
        return storedSession;
    }

    @Override
    @NotNull
    public Set<String> getAllClients(int bucketIndex) {
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        return ImmutableSet.copyOf(bucket.keySet());
    }

    @VisibleForTesting
    void removeWithTimestamp(@NotNull String clientId, int bucketIndex) {
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        PersistenceEntry<ClientSession> remove = bucket.remove(clientId);
        if (remove != null) {
            ClientSession clientSession = remove.getObject();
            if (ClientSessionMemoryLocalPersistence.isPersistent(clientSession) || clientSession.isConnected()) {
                this.sessionsCount.decrementAndGet();
            }
            this.removeWillReference(clientSession);
            this.currentMemorySize.addAndGet(-(remove.getEstimatedSize() + ObjectMemoryEstimation.stringSize(clientId)));
        }
    }

    @Override
    @ExecuteInSingleWriter
    @NotNull
    public Set<String> cleanUp(int bucketIndex) {
        ThreadPreConditions.startsWith("single-writer");
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        long currentTimeMillis = System.currentTimeMillis();
        Iterator<Map.Entry<String, PersistenceEntry<ClientSession>>> iterator = bucket.entrySet().iterator();
        ImmutableSet.Builder expiredClientIds = ImmutableSet.builder();
        while (iterator.hasNext()) {
            Map.Entry<String, PersistenceEntry<ClientSession>> entry = iterator.next();
            PersistenceEntry<ClientSession> storedEntry = entry.getValue();
            long timestamp = storedEntry.getTimestamp();
            ClientSession clientSession = storedEntry.getObject();
            long sessionExpiryInterval = clientSession.getSessionExpiryIntervalSec();
            if (!clientSession.isExpired(currentTimeMillis - timestamp)) continue;
            if (sessionExpiryInterval > 0L) {
                this.sessionsCount.decrementAndGet();
            }
            this.eventLog.clientSessionExpired(timestamp + sessionExpiryInterval * 1000L, entry.getKey());
            expiredClientIds.add((Object)entry.getKey());
            this.currentMemorySize.addAndGet(-(storedEntry.getEstimatedSize() + ObjectMemoryEstimation.stringSize(entry.getKey())));
            iterator.remove();
        }
        return expiredClientIds.build();
    }

    @Override
    @NotNull
    public Set<String> getDisconnectedClients(int bucketIndex) {
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        long currentTimeMillis = System.currentTimeMillis();
        return (Set)bucket.entrySet().stream().filter(entry -> !((ClientSession)((PersistenceEntry)entry.getValue()).getObject()).isConnected()).filter(entry -> ((ClientSession)((PersistenceEntry)entry.getValue()).getObject()).getSessionExpiryIntervalSec() > 0L).filter(entry -> !((ClientSession)((PersistenceEntry)entry.getValue()).getObject()).isExpired(currentTimeMillis - ((PersistenceEntry)entry.getValue()).getTimestamp())).map(Map.Entry::getKey).collect(ImmutableSet.toImmutableSet());
    }

    @Override
    public int getSessionsCount() {
        return this.sessionsCount.get();
    }

    @Override
    @ExecuteInSingleWriter
    public void setSessionExpiryInterval(@NotNull String clientId, long sessionExpiryInterval, int bucketIndex) {
        Preconditions.checkNotNull((Object)clientId, (Object)"Client Id must not be null");
        ThreadPreConditions.startsWith("single-writer");
        if (sessionExpiryInterval < 0L) {
            throw new InvalidSessionExpiryIntervalException("Invalid session expiry interval " + sessionExpiryInterval);
        }
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        bucket.compute(clientId, (ignored, storedSession) -> {
            if (storedSession == null) {
                throw NoSessionException.INSTANCE;
            }
            ClientSession clientSession = (ClientSession)storedSession.getObject();
            if (!clientSession.isConnected() && !ClientSessionMemoryLocalPersistence.isPersistent(clientSession)) {
                throw NoSessionException.INSTANCE;
            }
            clientSession.setSessionExpiryIntervalSec(sessionExpiryInterval);
            return new PersistenceEntry<ClientSession>(clientSession, storedSession.getTimestamp());
        });
    }

    @Override
    @NotNull
    public Map<String, PendingWillMessages.PendingWill> getPendingWills(int bucketIndex) {
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        return bucket.entrySet().stream().filter(entry -> !((ClientSession)((PersistenceEntry)entry.getValue()).getObject()).isConnected()).filter(entry -> ((ClientSession)((PersistenceEntry)entry.getValue()).getObject()).getWillPublish() != null).collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, entry -> {
            PersistenceEntry storedSession = (PersistenceEntry)entry.getValue();
            ClientSessionWill willPublish = ((ClientSession)storedSession.getObject()).getWillPublish();
            return new PendingWillMessages.PendingWill(Math.min(willPublish.getDelayInterval(), ((ClientSession)storedSession.getObject()).getSessionExpiryIntervalSec()), willPublish.getDelayInterval());
        }));
    }

    @Override
    @ExecuteInSingleWriter
    @Nullable
    public PersistenceEntry<ClientSession> deleteWill(@NotNull String clientId, int bucketIndex) {
        ThreadPreConditions.startsWith("single-writer");
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        PersistenceEntry persistenceEntry = bucket.computeIfPresent(clientId, (ignored, oldEntry) -> {
            ClientSession oldSession = (ClientSession)oldEntry.getObject();
            if (oldSession.isConnected()) {
                return oldEntry;
            }
            this.currentMemorySize.addAndGet(-oldEntry.getEstimatedSize());
            this.removeWillReference(oldSession);
            PersistenceEntry<ClientSession> newEntry = new PersistenceEntry<ClientSession>(oldSession.copyWithoutWill(), oldEntry.getTimestamp());
            this.currentMemorySize.addAndGet(newEntry.getEstimatedSize());
            return newEntry;
        });
        if (persistenceEntry == null) {
            return null;
        }
        ClientSession session = (ClientSession)persistenceEntry.getObject();
        if (session.isConnected()) {
            return null;
        }
        return new PersistenceEntry<ClientSession>(session.deepCopy(), persistenceEntry.getTimestamp());
    }

    @Override
    @NotNull
    public BucketChunkResult<Map<String, ClientSession>> getAllClientsChunk(int bucketIndex, @Nullable String ignored, int alsoIgnored) {
        long currentTimeMillis = System.currentTimeMillis();
        Map<String, PersistenceEntry<ClientSession>> bucket = this.getBucket(bucketIndex);
        Map<String, ClientSession> sessions = bucket.entrySet().stream().filter(entry -> {
            PersistenceEntry value = (PersistenceEntry)entry.getValue();
            return !((ClientSession)value.getObject()).isExpired(currentTimeMillis - value.getTimestamp());
        }).collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, entry -> ((ClientSession)((PersistenceEntry)entry.getValue()).getObject()).copyWithoutWill()));
        return new BucketChunkResult<Map<String, ClientSession>>(sessions, true, null, bucketIndex);
    }

    @Override
    @ExecuteInSingleWriter
    public void closeDB(int bucketIndex) {
        this.getBucket(bucketIndex).clear();
        this.sessionsCount.set(0);
        this.currentMemorySize.set(0L);
    }

    private void removeWillReference(@NotNull ClientSession clientSession) {
        ClientSessionWill willPublish = clientSession.getWillPublish();
        if (willPublish == null) {
            return;
        }
        this.metricsHolder.getStoredWillMessagesCount().dec();
        this.payloadPersistence.decrementReferenceCounter(willPublish.getPublishId());
    }

    private void loadWillPayload(@NotNull ClientSession clientSession) {
        this.loadWillPayload(clientSession, true);
    }

    private void loadWillPayload(@NotNull ClientSession clientSession, boolean setPayload) {
        ClientSessionWill willPublish = clientSession.getWillPublish();
        if (willPublish == null) {
            return;
        }
        if (willPublish.getPayload() != null) {
            return;
        }
        byte[] payload = this.payloadPersistence.get(willPublish.getPublishId());
        if (payload == null) {
            clientSession.setWillPublish(null);
            log.warn("Will Payload for payloadid {} not found", (Object)willPublish.getPublishId());
            return;
        }
        if (setPayload) {
            willPublish.getMqttWillPublish().setPayload(payload);
        }
    }

    private static boolean isPersistent(@NotNull ClientSession clientSession) {
        return clientSession.getSessionExpiryIntervalSec() > 0L;
    }
}

