/**
 * ******************************************************************************
 * Copyright (c) {2021} The original author or authors
 *
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License 2.0 which is available 
 * at http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: Apache-2.0 OR EPL-2.0
 ********************************************************************************/

package test.de.iip_ecosphere.platform.services;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import de.iip_ecosphere.platform.services.ArtifactDescriptor;
import de.iip_ecosphere.platform.services.ServiceDescriptor;
import de.iip_ecosphere.platform.services.ServiceFactory;
import de.iip_ecosphere.platform.services.ServiceManager;
import de.iip_ecosphere.platform.services.ServicesAas;
import de.iip_ecosphere.platform.services.ServicesAasClient;
import de.iip_ecosphere.platform.services.TypedDataConnectorDescriptor;
import de.iip_ecosphere.platform.services.environment.ServiceState;
import de.iip_ecosphere.platform.support.CollectionUtils;
import de.iip_ecosphere.platform.support.LifecycleHandler;
import de.iip_ecosphere.platform.support.Schema;
import de.iip_ecosphere.platform.support.Server;
import de.iip_ecosphere.platform.support.ServerAddress;
import de.iip_ecosphere.platform.support.aas.AasFactory;
import de.iip_ecosphere.platform.support.aas.AasPrintVisitor;
import de.iip_ecosphere.platform.support.aas.AasServer;
import de.iip_ecosphere.platform.support.aas.ProtocolServerBuilder;
import de.iip_ecosphere.platform.support.aas.ServerRecipe;
import de.iip_ecosphere.platform.support.iip_aas.AasPartRegistry;
import de.iip_ecosphere.platform.support.iip_aas.AbstractAasLifecycleDescriptor;
import de.iip_ecosphere.platform.support.iip_aas.ActiveAasBase;
import de.iip_ecosphere.platform.support.iip_aas.Id;
import de.iip_ecosphere.platform.support.json.JsonUtils;
import de.iip_ecosphere.platform.support.iip_aas.AasPartRegistry.AasSetup;
import de.iip_ecosphere.platform.support.iip_aas.ActiveAasBase.NotificationMode;
import test.de.iip_ecosphere.platform.transport.TestWithQpid;

/**
 * Tests the {@link ServicesAas}.
 * 
 * @author Holger Eichelberger, SSE
 */
public class ServicesAasTest extends TestWithQpid {

    private static Server qpid;

    /**
     * Initializes the test.
     */
    @BeforeClass
    public static void startup() {
        loadPlugins();
        ServerAddress broker = new ServerAddress(Schema.IGNORE);
        qpid = TestWithQpid.fromPlugin(broker);
        ServiceFactory.getTransport().setPort(broker.getPort());
        qpid.start();
    }
    
    /**
     * Shuts down the test.
     */
    @AfterClass
    public static void shutdown() {
        qpid.stop(true);
    }
    
    /**
     * Tests {@link ServicesAas}.
     * 
     * @throws IOException shall not occur
     * @throws ExecutionException shall not occur 
     * @throws URISyntaxException shall not occur
     */
    @Test
    public void testAas() throws IOException, ExecutionException, URISyntaxException {
        NotificationMode oldM = ActiveAasBase.setNotificationMode(NotificationMode.SYNCHRONOUS);
        Assert.assertTrue(AasPartRegistry.contributorClasses().contains(ServicesAas.class));
        AasSetup newSetup = AasSetup.createLocalEphemeralSetup();
        System.out.println("Registry: " + newSetup.getRegistryEndpoint().toUri());
        System.out.println("Servier: " + newSetup.getServerEndpoint().toUri());
        AasSetup oldSetup = AasPartRegistry.setAasSetup(newSetup);
        AasPartRegistry.AasBuildResult res = AasPartRegistry.build(c -> c instanceof ServicesAas);
        
        // active AAS require two server instances and a deployment
        ProtocolServerBuilder pBuilder = res.getProtocolServerBuilder();
        Server implServer = pBuilder.build();
        implServer.start();
        Server aasServer = AasPartRegistry.deploy(res.getAas()); 
        aasServer.start();
        AasPartRegistry.retrieveIipAas().accept(new AasPrintVisitor());
        ServicesAasClient client = new ServicesAasClient(Id.getDeviceIdAas(), "nonExApp");
        test(client);
        
        aasServer.stop(true);
        implServer.stop(true);
        AasPartRegistry.setAasSetup(oldSetup);
        ActiveAasBase.setNotificationMode(oldM);
    }

    /**
     * Tests the {@link ServicesAas} via the lifecycle descriptors.
     * 
     * @throws IOException shall not occur
     * @throws ExecutionException shall not occur 
     * @throws URISyntaxException shall not occur
     */
    @Test
    public void testLifecycle() throws IOException, ExecutionException, URISyntaxException {
        NotificationMode oldM = ActiveAasBase.setNotificationMode(NotificationMode.SYNCHRONOUS);
        boolean oldWaitIip = AbstractAasLifecycleDescriptor.setWaitForIipAas(false);
        AasSetup aasSetup = AasSetup.createLocalEphemeralSetup(null, false);
        AasSetup oldSetup = AasPartRegistry.setAasSetup(aasSetup);
        ServiceFactory.setAasSetup(aasSetup);

        ServerRecipe rcp = AasFactory.getInstance().createServerRecipe();
        Server registryServer = rcp
            .createRegistryServer(aasSetup, ServerRecipe.LocalPersistenceType.INMEMORY)
            .start();
        AasServer aasServer = rcp
            .createAasServer(aasSetup, ServerRecipe.LocalPersistenceType.INMEMORY)
            .start();

        LifecycleHandler.startup(new String[] {});

        ServicesAasClient client = new ServicesAasClient(Id.getDeviceIdAas());
        test(client);
        
        LifecycleHandler.shutdown();

        aasServer.stop(true);
        registryServer.stop(true);

        AbstractAasLifecycleDescriptor.setWaitForIipAas(oldWaitIip);
        AasPartRegistry.setAasSetup(oldSetup);
        ActiveAasBase.setNotificationMode(oldM);
    }
    
    /**
     * Tests the serivces AAS client.
     * 
     * @param client the client
     * @throws IOException shall not occur
     * @throws ExecutionException shall not occur 
     * @throws URISyntaxException shall not occur
     */
    private void test(ServicesAasClient client) throws IOException, ExecutionException, URISyntaxException {
        ServiceManager mgr = ServiceFactory.getServiceManager(); // for x-checking

        final URI dummy = new URI("file:///dummy");
        String aId = client.addArtifact(dummy);
        Assert.assertNotNull(aId);
        ArtifactDescriptor aDesc = mgr.getArtifact(aId);
        Assert.assertNotNull(aId);
        Assert.assertTrue(mgr.getArtifactIds().contains(aId));
        Assert.assertTrue(mgr.getArtifacts().contains(aDesc));

        Assert.assertEquals(aId, aDesc.getId());
        Assert.assertTrue(aDesc.getName().length() > 0);
        Assert.assertNotNull(aDesc.getUri());
        Assert.assertTrue(aDesc.getServiceIds().size() > 0);
        List<String> sIds = CollectionUtils.toList(aDesc.getServiceIds().iterator());
        String sId = sIds.get(0);
        Assert.assertNotNull(sId);
        ServiceDescriptor sDesc = aDesc.getService(sId);
        Assert.assertNotNull(sDesc);
        Assert.assertTrue(aDesc.getServiceIds().contains(sDesc.getId()));
        Assert.assertTrue(aDesc.getServices().contains(sDesc));
        Assert.assertEquals(ServiceState.AVAILABLE, client.getServiceState(sId));
        String[] ids = client.getServices(aDesc.getId(), false);
        Assert.assertEquals(aDesc.getServices().size(), ids.length);
        boolean foundAll = true;
        for (ServiceDescriptor s : aDesc.getServices()) {
            boolean found = false;
            for (int i = 0; !found && i < ids.length; i++) {
                found = ids[i].equals(s.getId());
            }
            foundAll &= found;
        }
        Assert.assertTrue(foundAll);
        Assert.assertNotNull(client.getArtifacts());
        Assert.assertNotNull(client.getRelations());
        Assert.assertNotNull(client.getServices());
        
        Predicate<TypedDataConnectorDescriptor> av = ServicesAas.createAvailabilityPredicate(1000, 200, false);
        Assert.assertFalse(av.test(sDesc.getOutputDataConnectors().get(0))); // not running/connected
        
        client.startService(sId);
        Assert.assertEquals(ServiceState.RUNNING, client.getServiceState(sId));
        AasPartRegistry.retrieveIipAas().accept(new AasPrintVisitor());
        
        Assert.assertEquals(1, client.getServiceInstanceCount(sId));
        
        client.passivateService(sId);
        Assert.assertEquals(ServiceState.PASSIVATED, client.getServiceState(sId));
        client.activateService(sId);
        Assert.assertEquals(ServiceState.RUNNING, client.getServiceState(sId));
        Map<String, String> vals = new HashMap<String, String>();
        vals.put("here", "1.23");
        vals.put("there", "{id:15, val:12}");
        client.reconfigureService(sId, vals);
        Assert.assertEquals(ServiceState.RUNNING, client.getServiceState(sId));
        client.setServiceState(sId, ServiceState.RUNNING); // no effect, just call
        Assert.assertTrue(av.test(sDesc.getOutputDataConnectors().get(0)));
        client.stopService(sId);
        Assert.assertEquals(ServiceState.STOPPED, client.getServiceState(sId));

        ServiceManagerTest.assertException(() -> mgr.cloneArtifact(aId, dummy));
        ServiceManagerTest.assertException(() -> mgr.migrateService(aId, "other"));
        ServiceManagerTest.assertException(() -> mgr.switchToService(aId, sId));
        mgr.updateService(aId, dummy);
        
        // test with multiple ids
        Map<String, String> options = new HashMap<>();
        options.put(ServicesAasClient.OPTION_ARGS, 
            JsonUtils.toJson(CollectionUtils.toList("-Dx.y=true", "-Dy.z=false")));
        options.put(ServicesAasClient.OPTION_PARAMS, "{\"service\":{\"p1\":\"v1\"}");
        client.startService(options, ids);
        assertOptions(mgr, options);
        client.stopService(ids);
        
        client.removeArtifact(aId);
        Assert.assertFalse(mgr.getArtifactIds().contains(aId));
        Assert.assertFalse(mgr.getArtifacts().contains(aDesc));
    }
    
    /**
     * Asserts received options on {@code mgr}.
     * 
     * @param mgr the manager instance
     * @param options the expected options
     */
    private void assertOptions(ServiceManager mgr, Map<String, String> options) {
        Assert.assertTrue(mgr instanceof MyServiceManager);
        Map<String, String> received = ((MyServiceManager) mgr).getLastOptions();
        if (options == null) {
            Assert.assertNull(received);
        } else {
            Assert.assertEquals(options, received);
        }
    }

}
