/**
 * ******************************************************************************
 * Copyright (c) {2021} The original author or authors
 *
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License 2.0 which is available 
 * at http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: Apache-2.0 OR EPL-2.0
 ********************************************************************************/

package test.de.iip_ecosphere.platform.support;

import java.util.ArrayList;
import java.util.List;

import org.junit.Assert;
import org.junit.Test;

import de.iip_ecosphere.platform.support.setup.CmdLine;

/**
 * Tests {@link CmdLine}.
 * 
 * @author Holger Eichelberger, SSE
 */
public class CmdLineTest {
    
    /**
     * Tests {@link CmdLine#parseToArgs(String, java.util.List)}.
     */
    @Test
    public void testParseToArgs() {
        List<String> result = new ArrayList<String>();

        CmdLine.parseToArgs("", result);
        Assert.assertTrue(result.size() == 0);
        
        CmdLine.parseToArgs("--a=0", result);
        Assert.assertTrue(result.size() == 1);
        Assert.assertEquals("--a=0", result.get(0));
        result.clear();

        CmdLine.parseToArgs("--a=0 --b=5", result);
        Assert.assertTrue(result.size() == 2);
        Assert.assertEquals("--a=0", result.get(0));
        Assert.assertEquals("--b=5", result.get(1));
        result.clear();

        CmdLine.parseToArgs("\"--a=0 --b=5\"", result);
        Assert.assertTrue(result.size() == 1);
        Assert.assertEquals("\"--a=0 --b=5\"", result.get(0));
        result.clear();

        CmdLine.parseToArgs("--a=0 --c=\"x y\" --b=5", result);
        Assert.assertTrue(result.size() == 3);
        Assert.assertEquals("--a=0", result.get(0));
        Assert.assertEquals("--c=\"x y\"", result.get(1));
        Assert.assertEquals("--b=5", result.get(2));
        result.clear();
    }
    
    /**
     * Tests the getArg functions.
     */
    @Test
    public void testGetArg() {
        String[] args = {"--bli=5", "--bla=6", "--xyz=String", "whatever", "--bVal=true"};
        Assert.assertEquals("6", CmdLine.getArg(args, "bla", ""));
        Assert.assertEquals("5", CmdLine.getArg(args, "bli", ""));
        Assert.assertEquals("String", CmdLine.getArg(args, "xyz", ""));

        Assert.assertEquals(5, CmdLine.getIntArg(args, "bli", 0));
        Assert.assertEquals(0, CmdLine.getIntArg(args, "blii", 0));
        
        Assert.assertEquals(true, CmdLine.getBooleanArg(args, "bVal", false));
        Assert.assertEquals(false, CmdLine.getBooleanArg(args, "xVal", false));
        Assert.assertEquals(false, CmdLine.getBooleanArg(args, "bla", false));
    }

    /**
     * Tests {@link CmdLine#toArgs(String)}.
     */
    @Test
    public void testToArgs() {
        String[] tmp = CmdLine.toArgs("");
        Assert.assertNotNull(tmp);
        Assert.assertEquals(0, tmp.length);

        tmp = CmdLine.toArgs("--iip.app.p=5 --iip.app.w=7 --transport=5");
        Assert.assertNotNull(tmp);
        Assert.assertEquals(3, tmp.length);
        Assert.assertEquals("--iip.app.p=5", tmp[0]);
        Assert.assertEquals("--iip.app.w=7", tmp[1]);
        Assert.assertEquals("--transport=5", tmp[2]);

        tmp = CmdLine.toArgs("--iip.app.p='5' --iip.app.w=\"7\" --transport=5");
        Assert.assertNotNull(tmp);
        Assert.assertEquals(3, tmp.length);
        Assert.assertEquals("--iip.app.p='5'", tmp[0]);
        Assert.assertEquals("--iip.app.w=\"7\"", tmp[1]);
        Assert.assertEquals("--transport=5", tmp[2]);
    }
    
    /**
     * Tests {@link CmdLine#composeArgument(String, Object)}.
     */
    @Test
    public void testComposeArgument() {
        Assert.assertEquals("--iip.app.p=5", CmdLine.composeArgument("iip.app.p", "5"));
    }
    
    /**
     * Tests {@link CmdLine#hasArgument(String[], String, boolean, boolean)}.
     */
    @Test
    public void testHasArgument() {
        final String[] arg = {"--repo", "--online=true"};
        
        Assert.assertFalse(CmdLine.hasArgument(new String[] {}, "test", false, true));
        Assert.assertFalse(CmdLine.hasArgument(new String[] {}, "test", true, true));
        Assert.assertTrue(CmdLine.hasArgument(arg, "repo", true, true));
        Assert.assertFalse(CmdLine.hasArgument(arg, "repo", false, true));
        Assert.assertTrue(CmdLine.hasArgument(arg, "online", true, true));
        Assert.assertTrue(CmdLine.hasArgument(arg, "online", false, true));
        Assert.assertFalse(CmdLine.hasArgument(arg, "online", true, false));
    }

    /**
     * Tests {@link CmdLine#getBooleanArgNoVal(String[], String, boolean)}.
     */
    @Test
    public void testGetBooleanArgNoVal() {
        Assert.assertFalse(CmdLine.getBooleanArgNoVal(new String[] {"--repo1"}, "repo", false));
        Assert.assertTrue(CmdLine.getBooleanArgNoVal(new String[] {"--repo"}, "repo", false));
        Assert.assertTrue(CmdLine.getBooleanArgNoVal(new String[] {"--repo=true"}, "repo", false));
        Assert.assertFalse(CmdLine.getBooleanArgNoVal(new String[] {"--repo=false"}, "repo", false));
    }

    /**
     * Tests {@link CmdLine#extractArgNames(String[])}.
     */
    @Test
    public void testExtractArgs() {
        String[] names = CmdLine.extractArgNames(new String[] {});
        Assert.assertNotNull(names);
        Assert.assertEquals(0, names.length);

        names = CmdLine.extractArgNames(new String[] {"--repo1"});
        Assert.assertNotNull(names);
        Assert.assertArrayEquals(new String[] {"repo1"}, names);

        names = CmdLine.extractArgNames(new String[] {"--repo1=myRepo"});
        Assert.assertNotNull(names);
        Assert.assertArrayEquals(new String[] {"repo1"}, names);

        names = CmdLine.extractArgNames(new String[] {"--repo1=myRepo", "--repo1.active=true", "-Djava=true"});
        Assert.assertNotNull(names);
        Assert.assertArrayEquals(new String[] {"repo1", "repo1.active"}, names);
    }

}
