package de.uni_hildesheim.sse.trans.in.rsf;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.in.AbstractReader;
import de.uni_hildesheim.sse.trans.in.ParserException;

/**
 * Represents a condition read from a "Depends" line in a .rsf file.
 * The format looks like this: <br />
 * <code>
 * Depends <i>variable</i> <i>condition</i>
 * </code> <br />
 * <i>variable</i> can only be true, if <i>condition</i> is true. <br />
 * {@link RSFCondition#getConstraintSyntaxTreeHack(AbstractReader, String)} will convert it into: <br />
 * <code>
 * (<i>variable</i> AND <i>condition</i>) OR (NOT <i>variable</i> AND NOT <i>condition</i>)
 * </code>
 * 
 * @author Adam Krafczyk
 */
class RSFDependsCondition extends RSFCondition {

    private String variable;
    private String condition;
    
    /**
     * Creates a condition with the given variable and condition as read from a "Depends" line in a .rsf file.
     * @param variable The name of the variable
     * @param condition The condition as read from the .rsf file
     */
    RSFDependsCondition(String variable, String condition) {
        this.variable = variable;
        this.condition = condition;
        // TODO error handling if only on is true?
        if (this.condition.startsWith("\"") && this.condition.endsWith("\"")) {
            this.condition = this.condition.substring(1, this.condition.length() - 1);
        }
    }
    
    /**
     * {@inheritDoc}
     */
    ConstraintSyntaxTree toConstraintSyntaxTree(AbstractReader reader) throws ParserException {
        DecisionVariableDeclaration varDecl = reader.getVariable(variable);
        Variable var = varPool.obtainVariable(varDecl);
        OCLFeatureCall notVar = new OCLFeatureCall(var, OclKeyWords.NOT);
        
        ConstraintSyntaxTree condition = getConstraintSyntaxTree(reader, this.condition);
        OCLFeatureCall notCondition = new OCLFeatureCall(condition, OclKeyWords.NOT);
        
        OCLFeatureCall leftANDCall = new OCLFeatureCall(var, OclKeyWords.AND, condition);
        OCLFeatureCall rightANDCall = new OCLFeatureCall(notVar , OclKeyWords.AND, notCondition);
        
        return new OCLFeatureCall(leftANDCall, OclKeyWords.OR, rightANDCall);
    }
    
}
