package de.uni_hildesheim.sse.trans.in.rsf;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.in.AbstractReader;
import de.uni_hildesheim.sse.trans.in.ParserException;

/**
 * Represents a condition read from a "ItemSelects" line in a .rsf file. <br />
 * The format looks like this: <br />
 * <code>
 * ItemSelects <i>variable</i> <i>selectedVaraible</i> <i>condition</i>
 * </code> <br />
 * <i>selectedVaraible</i> has to be set to true, if <i>variable</i> and <i>condition</i> are true. <br />
 * {@link RSFItemSelectsCondition#getConstraintSyntaxTreeHack(AbstractReader, String)} will convert it to: <br />
 * <code>
 * (NOT <i>variable</i> OR NOT <i>condition</i>) OR (<i>variable</i> AND <i>selectedVaraible</i> AND <i>condition</i>)
 * </code>
 * @author Adam Krafczyk
 */
class RSFItemSelectsCondition extends RSFCondition {

    private String variable;
    private String selectedVariable;
    private String condition;
    
    /**
     * Creates a condition with the given variable and condition as read from a "ItemSelects" line in a .rsf file.
     * @param variable The name of the variable
     * @param selectedVaraible The name of the variable that is selected
     * @param condition The condition as read from the .rsf file
     */
    RSFItemSelectsCondition(String variable, String selectedVaraible, String condition) {
        this.variable = variable;
        this.selectedVariable = selectedVaraible;
        // TODO error handling if only on is true?
        if (this.selectedVariable.startsWith("\"") && this.selectedVariable.endsWith("\"")) {
            this.selectedVariable = this.selectedVariable.substring(1, this.selectedVariable.length() - 1);
        }
        this.condition = condition;
        // TODO error handling if only on is true?
        if (this.condition.startsWith("\"") && this.condition.endsWith("\"")) {
            this.condition = this.condition.substring(1, this.condition.length() - 1);
        }
    }
    
    @Override
    ConstraintSyntaxTree toConstraintSyntaxTree(AbstractReader reader) throws ParserException {
        DecisionVariableDeclaration varDecl = reader.getVariable(variable);
        Variable var = varPool.obtainVariable(varDecl);
        OCLFeatureCall notVar = new OCLFeatureCall(var, OclKeyWords.NOT);
        
        // TODO can selectedVariable be a condition?
        DecisionVariableDeclaration selectedVarDecl = reader.getVariable(selectedVariable);
        Variable selectedVar = varPool.obtainVariable(selectedVarDecl);
        
        ConstraintSyntaxTree condition = getConstraintSyntaxTree(reader, this.condition, variable);
        OCLFeatureCall notCondition = new OCLFeatureCall(condition, OclKeyWords.NOT);
        
        OCLFeatureCall leftOR = new OCLFeatureCall(notVar, OclKeyWords.OR, notCondition);
        OCLFeatureCall rightAND1 = new OCLFeatureCall(var, OclKeyWords.AND, selectedVar);
        OCLFeatureCall rightAND2 = new OCLFeatureCall(rightAND1, OclKeyWords.AND, condition);
        
        return new OCLFeatureCall(leftOR, OclKeyWords.OR, rightAND2);
    }
    
}
