package de.uni_hildesheim.sse.trans.in.rsf;

import java.util.ArrayList;
import java.util.List;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.varModel.datatypes.Enum;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.in.AbstractReader;
import de.uni_hildesheim.sse.trans.in.ParserException;

/**
 * Contains a Choice and the ChoiceItems.
 * {@link RSFChoice#toPureBooleanConstraintSyntaxTree(RSFReader)} assures, that only one variable of the choice is
 * selected.
 * 
 * @author Adam Krafczyk
 */
class RSFChoice extends RSFCondition {
    
    private RSFItem choiceItem;
    private List<RSFItem> choiceItems;
    
    private List<String> choiceItemConditions;
    private String choiceCondition;
    
    /**
     * Creates a new Choice with no ChoiceItems.
     * @param choiceItem The name of the Choice.
     */
    RSFChoice(RSFItem choiceItem) {
        this.choiceItem = choiceItem;
        choiceItems = new ArrayList<RSFItem>();
        choiceItemConditions = new ArrayList<String>();
    }
    
    /**
     * Adds a ChoiceItem to the Choice.
     * @param item The name of the ChoiceItem.
     */
    void addChoiceItem(RSFItem item) {
        choiceItems.add(item);
    }
    
    /**
     * Adds a constraint of a choice item of this choice.
     * @param condition The condition as read from a depends line in the .rsf file.
     */
    void addChoiceItemDepends(String condition) {
        // TODO error handling if only one is true?
        if (condition.startsWith("\"") && condition.endsWith("\"")) {
            condition = condition.substring(1, condition.length() - 1);
        }
        
        choiceItemConditions.add(condition);
    }
    
    /**
     * Adds a constraint the choice.
     * @param condition A condition as read from a depends line in the .rsf file.
     */
    void addChoiceCondition(String condition) {
        // TODO error handling if only one is true?
        if (condition.startsWith("\"") && condition.endsWith("\"")) {
            condition = condition.substring(1, condition.length() - 1);
        }
        
        if (choiceCondition == null) {
            choiceCondition = condition;
        } else {
            choiceCondition = "(" + choiceCondition + ") || (" + condition + ")";
        }
    }
    
    /**
     * Creates and returns the additional depends condition that is needed to ensure that the model is not broken
     * when all choice items can't fulfill their conditions.
     * @return A {@link RSFDependsCondition} or <code>null</code> if no conditions have been added.
     */
    RSFDependsCondition getAdditionalDependsCondition() {
        RSFDependsCondition result = null;
        
        if (choiceItemConditions.size() > 0) {
            StringBuffer condition = new StringBuffer();
            if (choiceCondition != null) {
                condition.append("\"(");
                condition.append(choiceCondition);
                condition.append(") && (");
            } else {
                condition.append("\"(");
            }
            
            for (int i = 0; i < choiceItemConditions.size(); i++) {
                String choiceItemCondition = choiceItemConditions.get(i);
                boolean addBrackets = choiceItemCondition.contains("||") || choiceItemCondition.contains("&&");
                if (addBrackets) {
                    condition.append("(");
                }
                condition.append(choiceItemCondition);
                if (addBrackets) {
                    condition.append(")");
                }
                if (i != choiceItemConditions.size() - 1) {
                    condition.append(" || ");
                }
            }
            condition.append(")\"");
            result = new RSFDependsCondition(choiceItem, condition.toString());
        }
        
        return result;
    }
    
    @Override
    List<ConstraintSyntaxTree> toPureBooleanConstraintSyntaxTree(RSFReader reader) throws ParserException {
        //ConstraintSyntaxTree[] trees = new ConstraintSyntaxTree[choiceItems.size()];
        List<ConstraintSyntaxTree> trees = new ArrayList<ConstraintSyntaxTree>();
        
        ConstraintSyntaxTree choiceVar = getSelectedVariable(choiceItem, reader);
        ConstraintSyntaxTree notChoiceVar = getUnselectedVariable(choiceItem, reader);
        
        // Create one tree with all variables OR'd together (to make sure that at least one is selected if choiceVar
        //   is true)
        ConstraintSyntaxTree allItemsTree = notChoiceVar;
        
        // And for each unique combination of choiceItems:
        for (int i = 0; i < choiceItems.size(); i++) {
            ConstraintSyntaxTree var1 = getSelectedVariable(choiceItems.get(i), reader);
            ConstraintSyntaxTree notVar1 = getUnselectedVariable(choiceItems.get(i), reader);
            
            for (int j = i + 1; j < choiceItems.size(); j++) {
                // Add the 2 variables negated and OR'd together: (to make sure that only one or less vars are selected)
                ConstraintSyntaxTree notVar2 = getUnselectedVariable(choiceItems.get(j), reader);
                
                trees.add(new OCLFeatureCall(notVar1, OclKeyWords.OR, notVar2));
            }
            
            allItemsTree = new OCLFeatureCall(allItemsTree, OclKeyWords.OR, var1);
            
            // Add choiceVar or not var1 (to make sure that no variables are selected if choiceVar is false)
            trees.add(new OCLFeatureCall(choiceVar, OclKeyWords.OR, notVar1));
        }
        
        trees.add(allItemsTree);
        
        return trees;
    }

    @Override
    ConstraintSyntaxTree toNotBooleanConstraintSyntaxTree(AbstractReader reader, Enum tristate) throws ParserException {
        // TODO implement
        return null;
    }
    
}
