package de.uni_hildesheim.sse.trans.in.rsf;

import java.util.ArrayList;
import java.util.List;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.varModel.datatypes.Enum;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.in.AbstractReader;
import de.uni_hildesheim.sse.trans.in.ParserException;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Collects all default values with conditions and all prompt conditions of an
 * item and then constructs constraints from them.
 * 
 * @author Adam Krafczyk
 */
class RSFDefaultAndPromptCondition extends RSFCondition {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(
            RSFDefaultAndPromptCondition.class, "RSFDefaultAndPromptCondition");

    private RSFItem item;
    private List<String> promptConditions;
    private List<DefaultValue> defaultValues;
    
    /**
     * Creates a new {@link RSFDefaultAndPromptCondition} for the given item.
     * @param item The item which Default and Promp conditions will be added.
     */
    RSFDefaultAndPromptCondition(RSFItem item) {
        this.item = item;
        promptConditions = new ArrayList<String>();
        defaultValues = new ArrayList<DefaultValue>();
    }
    
    /**
     * Adds a prompt condition for the given item.
     * @param condition The condition as read from the .rsf file.
     */
    void addPromptCondition(String condition) {
        promptConditions.add(condition);
    }
    
    /**
     * Adds a default value with a condition for the item.
     * @param value The default value as read from the .rsf file.
     * @param condition The condition as read from the .rsf file.
     */
    void addDefaultCondition(String value, String condition) {
        defaultValues.add(new DefaultValue(value, condition));
    }
    
    @Override
    List<ConstraintSyntaxTree> toPureBooleanConstraintSyntaxTree(RSFReader reader) throws ParserException {
        List<ConstraintSyntaxTree> trees = new ArrayList<ConstraintSyntaxTree>();
        
        List<ConstraintSyntaxTree> promptConditionTrees = new ArrayList<ConstraintSyntaxTree>();
        
        boolean oneAlwaysTrue = false;
        for (String condition : promptConditions) {
            ConstraintSyntaxTree conditionTree = getPureBooleanConstraintSyntaxTree(
                    reader, condition, item.getName());
            if (conditionTree == null) {
                oneAlwaysTrue = true;
                break;
            }
            promptConditionTrees.add(conditionTree);
        }
        
        if (!oneAlwaysTrue) {
            ConstraintSyntaxTree prependCondition = null;
            if (promptConditionTrees.size() > 0) {
                prependCondition = promptConditionTrees.get(0);
                for (int i = 1; i < promptConditionTrees.size(); i++) {
                    prependCondition = new OCLFeatureCall(
                            prependCondition, OclKeyWords.OR, promptConditionTrees.get(i));
                }
            }
            
            
            for (DefaultValue defaultValue : defaultValues) {
                ConstraintSyntaxTree condition = getPureBooleanConstraintSyntaxTree(
                        reader, defaultValue.condition, item.getName());
                if (condition != null) {
                    condition = new OCLFeatureCall(condition, OclKeyWords.NOT);
                }
                
                boolean wasConstant = false;
                if (defaultValue.value.startsWith("'") && defaultValue.value.endsWith("'")) {
                    // value is constant
                    ConstraintSyntaxTree tree = null;
                    switch (item.getDatatype()) {
                    case HEX:
                    case INTEGER:
                    case STRING:
                        tree = getVariable(
                                reader, item.getName() + "=" + defaultValue.value, null);
                        wasConstant = true;
                        break;
                    
                    default:
                        tree = null;
                        if (defaultValue.value.equalsIgnoreCase("'y'")) {
                            tree = getSelectedVariable(item, reader);
                            wasConstant = true;
                        } else if (defaultValue.value.equalsIgnoreCase("'n'")
                                || defaultValue.value.equalsIgnoreCase("'no'")) {
                            tree = getUnselectedVariable(item, reader);
                            wasConstant = true;
                        } else if (defaultValue.value.equalsIgnoreCase("'m'")) {
                            if (!item.getDatatype().equals(Datatype.TRISTATE)) {
                                LOGGER.error("'m' as default value for non-tristate variable " + item.getName()
                                        + " (skipping)");
                                continue;
                            }
                            tree = varPool.obtainVariable(reader.getVariable(item.getName() + "_MODULE"));
                            wasConstant = true;
                        }
                        break;
                    }
                    
                    if (wasConstant) {
                        if (condition != null) {
                            tree = new OCLFeatureCall(tree, OclKeyWords.OR, condition);
                        }
                        if (prependCondition != null) {
                            tree = new OCLFeatureCall(tree, OclKeyWords.OR, prependCondition);
                        }
                        trees.add(tree);
                    }
                    
                }
                
                if (!wasConstant) {
                    // value is constraint
                    ConstraintSyntaxTree valueConstraint =
                            getPureBooleanConstraintSyntaxTree(reader, defaultValue.value, item.getName());
                    ConstraintSyntaxTree notValueConstraint = new OCLFeatureCall(valueConstraint, OclKeyWords.NOT);
                    
                    ConstraintSyntaxTree var = getSelectedVariable(item, reader);
                    ConstraintSyntaxTree notvar = getUnselectedVariable(item, reader);
                    
                    ConstraintSyntaxTree tree1 = new OCLFeatureCall(notvar, OclKeyWords.OR, valueConstraint);
                    ConstraintSyntaxTree tree2 = new OCLFeatureCall(var, OclKeyWords.OR, notValueConstraint);
                    
                    if (condition != null) {
                        tree1 = new OCLFeatureCall(tree1, OclKeyWords.OR, condition);
                        tree2 = new OCLFeatureCall(tree2, OclKeyWords.OR, condition);
                    }
                    if (prependCondition != null) {
                        tree1 = new OCLFeatureCall(tree1, OclKeyWords.OR, prependCondition);
                        tree2 = new OCLFeatureCall(tree2, OclKeyWords.OR, prependCondition);
                    }
                    
                    trees.add(tree1);
                    trees.add(tree2);
                }
                
                
            }
        }
        
        return trees;
    }

    @Override
    ConstraintSyntaxTree toNotBooleanConstraintSyntaxTree(AbstractReader reader, Enum tristate) throws ParserException {
        // TODO Auto-generated method stub
        return null;
    }
    
    /**
     * Stores both, the default value and a the corresponding condition.
     */
    private static class DefaultValue {
        private String value;
        private String condition;
        
        /**
         * Creates a {@link DefaultValue} with value and conidition as read from
         * the .rsf file.
         * @param value The value.
         * @param condition The condition.
         */
        DefaultValue(String value, String condition) {
            this.value = value;
            this.condition = condition;
        }
    }

}
