package de.uni_hildesheim.sse.trans.in.rsf;

import java.util.ArrayList;
import java.util.List;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.in.ParserException;

/**
 * Represents a condition read from a "Depends" line in a .rsf file.
 * @see <a href="https://projects.sse.uni-hildesheim.de/agilo/ModelTranslator/
 *wiki/Specification/RSF2DIMACS#TranslationofdependsConstraints">Specifikation</a>
 *
 * @author Adam Krafczyk
 */
class RSFDependsCondition extends RSFCondition {
//    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(RSFDependsCondition.class,
//            "RSFDependsCondition");

    private RSFItem variable;
    private List<String> conditions;
    
    /**
     * Creates a condition with the given variable and condition as read from a "Depends" line in a .rsf file.
     * @param variable The name of the variable
     * @param condition The condition as read from the .rsf file
     */
    RSFDependsCondition(RSFItem variable, String condition) {
        this.variable = variable;
        conditions = new ArrayList<String>();
        addCondition(condition);
    }
    
    /**
     * Adds another condition with the same base variable.
     * @param condition The other condition as read from another Depends variable line.
     */
    void addCondition(String condition) {
        // TODO error handling if only one is true?
        if (condition.startsWith("\"") && condition.endsWith("\"")) {
            condition = condition.substring(1, condition.length() - 1);
        }
        conditions.add(condition);
    }
    
    @Override
    List<ConstraintSyntaxTree> toPureBooleanConstraintSyntaxTree(RSFReader reader) throws ParserException {
        
        List<ConstraintSyntaxTree> conditionTrees = new ArrayList<ConstraintSyntaxTree>();
        
        for (String conditionString : conditions) {
            conditionTrees.add(getPureBooleanConstraintSyntaxTree(reader, conditionString, variable.getName()));
        }
        
        List<ConstraintSyntaxTree> result  = new ArrayList<ConstraintSyntaxTree>();
        
        boolean hasAtLeastOneNotNull = false;
        for (ConstraintSyntaxTree tree : conditionTrees) {
            if (tree != null) {
                hasAtLeastOneNotNull = true;
                break;
            }
        }
        
        if (hasAtLeastOneNotNull) {
            
            if (variable.getDatatype().equals(Datatype.TRISTATE)) {
                
                ConstraintSyntaxTree varTree = getUnselectedVariable(variable, reader);
                
                DecisionVariableDeclaration moduleDecl = reader.getVariable(variable.getName() + "_MODULE");
                ConstraintSyntaxTree moduleTree = new OCLFeatureCall(varPool.obtainVariable(moduleDecl),
                        OclKeyWords.NOT);
                
                for (ConstraintSyntaxTree conditionTree : conditionTrees) {
                    if (conditionTree != null) {
                        varTree = new OCLFeatureCall(varTree, OclKeyWords.OR, conditionTree);
                        moduleTree = new OCLFeatureCall(moduleTree, OclKeyWords.OR, conditionTree);
                    }
                }
                
                result.add(varTree);
                result.add(moduleTree);
                
            } else {
                
                ConstraintSyntaxTree tree = getUnselectedVariable(variable, reader);
                
                for (ConstraintSyntaxTree conditionTree : conditionTrees) {
                    if (conditionTree != null) {
                        tree = new OCLFeatureCall(tree, OclKeyWords.OR, conditionTree);
                    }
                }
                
                result.add(tree);
            }
        }
        
        return result;
    }
    
    @Override
    List<ConstraintSyntaxTree> toNotBooleanConstraintSyntaxTree(RSFReader reader) {
        List<ConstraintSyntaxTree> trees = new ArrayList<ConstraintSyntaxTree>();
        
        List<ConstraintSyntaxTree> conditionTrees = new ArrayList<ConstraintSyntaxTree>();
        
        for (String condition :  conditions) {
            ConstraintSyntaxTree conditionTree = getNotBooleanConstraintSyntaxTree(reader, condition,
                    variable.getName());
            if (conditionTree != null) {
                conditionTrees.add(conditionTree);
            }
        }
        
        ConstraintSyntaxTree notVar = null;
        Variable var = varPool.obtainVariable(reader.getVariable(variable.getName()));
        switch (variable.getDatatype()) {
        case BOOLEAN:
            notVar = new OCLFeatureCall(var, OclKeyWords.NOT);
            break;
        case TRISTATE:
            notVar = new OCLFeatureCall(var, OclKeyWords.EQUALS, RSFReader.TRI_UNDEFINED);
            break;
        case STRING:
        case HEX:
        case INTEGER:
            notVar = new OCLFeatureCall(var, OclKeyWords.IS_EMPTY); // TODO
            break;
            
        default:
            // can't happen
            break;
        }
        
        for (ConstraintSyntaxTree conditionTree : conditionTrees) {
            ConstraintSyntaxTree notCondition = new OCLFeatureCall(conditionTree, OclKeyWords.NOT);
            trees.add(new OCLFeatureCall(notCondition, OclKeyWords.IMPLIES, notVar));
        }
        
        return trees;
    }
    
}
