package de.uni_hildesheim.sse.model_extender.in;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.uni_hildesheim.sse.model.cst.CSTSemanticException;
import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.VariablePool;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.BooleanType;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;

/**
 * Loads a model from a .dimacs file.
 *  
 * @author Adam Krafczyk
 */
public class DimacsReader implements Closeable {

    private static final Pattern VARIABLE_PATTERN = Pattern.compile("^c ([0-9]+) ([\\S]+)$");
    
    private static final Pattern META_PATTERN = Pattern.compile("^p CNF ([0-9]+) ([0-9]+)$");
    
    private static final Pattern CONSTRAINT_PATTERN = Pattern.compile("^([\\x2D]?+[0-9]+[ ]?+)+0$");
    
    private BufferedReader in;
    
    private Project model;
    
    private Map<Integer, DecisionVariableDeclaration> variables;
    
    private VariablePool varPool;
    
    private int numDeclaredVariables;
    
    private int numDeclaredConstraints;
    
    /**
     * Creates a new {@link DimacsReader} for the given file.
     * 
     * @param file The file to read the model from.
     * 
     * @throws FileNotFoundException If the specified file can't be opened for reading.
     */
    public DimacsReader(File file) throws FileNotFoundException {
        in = new BufferedReader(new FileReader(file));
    }

    @Override
    public void close() throws IOException {
        in.close();
    }
    
    /**
     * Reads the model.
     * 
     * @return The model read from the .dimacs file.
     * 
     * @throws IOException If reading the file throws an {@link IOException}.
     * @throws MalformedFileException If the file is not correctly formatted.
     */
    public Project getModel() throws IOException, MalformedFileException {
        /*
         * Initialise variables
         */
        model = new Project("Dimacs Project");
        variables = new HashMap<Integer, DecisionVariableDeclaration>();
        varPool = new VariablePool();
        numDeclaredVariables = -1;
        numDeclaredConstraints = -1;
        
        readVariableDeclarations();
        
        readConstraints();
        
        /*
         * Deinitialize variables
         */
        Project modelCpy = model;
        model = null;
        variables = null;
        varPool = null;
        numDeclaredVariables = -1;
        numDeclaredConstraints = -1;
        
        return modelCpy; 
    }

    /**
     * Reads variable declarations and the meta line until the meta line is found.
     * 
     * @throws IOException If reading the file throws an {@link IOException}.
     * @throws MalformedFileException If the file is not correctly formatted.
     */
    private void readVariableDeclarations() throws IOException, MalformedFileException {
        String line;
        
        //Read variables until meta line is found.
        boolean foundMetaLine = false;
        while (!foundMetaLine) {
            line = in.readLine();
            
            if (line == null) {
                throw new MalformedFileException("No meta line found");
            }
            
            line = line.trim();
            
            boolean isVariable = parseVariableLine(line);
            
            if (!isVariable) {
                foundMetaLine = parseMetaLine(line);
            }
            
            if (!isVariable && !foundMetaLine && !line.startsWith("c") && line.length() > 0) {
                throw new MalformedFileException("Found invalid line in file: " + line);
            }
        }
        
        // Check if number of variables declared in meta line matches number of found variables
        int numFoundVariables = variables.entrySet().size();
        if (numFoundVariables != numDeclaredVariables) {
            StringBuilder message = new StringBuilder();
            message.append("Number of found variables is not equal to specified number in meta line:\n");
            message.append("\tSpecified: ").append(numDeclaredVariables).append("\n");
            message.append("\tFound: ").append(numFoundVariables).append("\n");
            
            for (int i = 1; i <= Math.max(numFoundVariables, numDeclaredVariables); i++) {
                if (i > numDeclaredVariables) {
                    message.append("Undeclared variable: ").append(i).append(" ")
                        .append(variables.get(i).getName()).append("\n");
                }
                if (!variables.containsKey(i)) {
                    message.append("Not found variable ").append(i).append("\n");
                }
            }
            
            throw new MalformedFileException(message.toString());
        }
    }

    /**
     * Reads constraints from the file.
     * 
     * @throws IOException If reading the file throws an {@link IOException}.
     * @throws MalformedFileException If the file is not correctly formatted.
     */
    private void readConstraints() throws IOException, MalformedFileException {
        // Read Constraints
        String line;
        int numParsedConstriants = 0;
        while ((line = in.readLine()) != null) {
            line = line.trim();
            if (!line.startsWith("c") && line.length() > 0) {
                parseConstraintLine(line);
                numParsedConstriants++;
            }
        }
        
        // Check if number of constraints declared in meta line matches number of found constraints
        if (numParsedConstriants != numDeclaredConstraints) {
            throw new MalformedFileException("Number of found constraints (" + numParsedConstriants
                    + ") is not equal to number of specified constraints in meta line ("
                    + numDeclaredConstraints + ")");
        }
        
        // Add variables to project
        for (DecisionVariableDeclaration var : variables.values()) {
            model.add(var);
        }
    }
    
    /**
     * Attempts to parse the line as a variable declaration.
     * 
     * @param line The line to parse.
     * 
     * @return <code>true</code> if the line is a valid variable declaration.
     */
    private boolean parseVariableLine(String line) {
        Matcher matcher = VARIABLE_PATTERN.matcher(line);
        
        boolean isVariableLine = false;
        
        if (matcher.matches()) {
            isVariableLine = true;
            
            try {
                int num = Integer.parseInt(matcher.group(1));
                String variableName = matcher.group(2);
                
                DecisionVariableDeclaration variable =
                        new DecisionVariableDeclaration(variableName, BooleanType.TYPE, model);
                
                variables.put(num, variable);
                
            } catch (NumberFormatException e) {
                isVariableLine = false;
            }
        }
        
        return isVariableLine;
    }
    
    /**
     * Attempts to parse the line as the meta line ("p CNF ...").
     * 
     * @param line The line to parse.
     * 
     * @return <code>true</code> if the line is a valid meta line.
     */
    private boolean parseMetaLine(String line) {
        Matcher matcher = META_PATTERN.matcher(line);
        
        boolean isMetaLine = false;
        
        if (matcher.matches()) {
            try {
                numDeclaredVariables = Integer.parseInt(matcher.group(1));
                numDeclaredConstraints = Integer.parseInt(matcher.group(2));
                isMetaLine = true;
            } catch (NumberFormatException e) {
                
            }
        }
        
        return isMetaLine;
    }
    
    /**
     * Attempts to parse the line as a constraint definition.
     * 
     * @param line The line to parse.
     * 
     * @throws MalformedFileException If the line is not a valid constraint.
     */
    private void parseConstraintLine(String line) throws MalformedFileException {
        // Check if it's a valid constraint line
        Matcher matcher = CONSTRAINT_PATTERN.matcher(line);
        if (!matcher.matches()) {
            throw new MalformedFileException("Found invalid constraint: " + line);
        }
        
        String[] parts = line.split(" ");
        
        // Construct CST
        ConstraintSyntaxTree tree = getVariable(parts[0]);
        
        for (int i = 1; i < parts.length - 1; i++) {
            tree = new OCLFeatureCall(tree, OclKeyWords.OR, getVariable(parts[i]));
        }
        
        // Add Constraint to project
        Constraint constraint = new Constraint(model);
        try {
            constraint.setConsSyntax(tree);
            model.add(constraint);
        } catch (CSTSemanticException e) {
            e.printStackTrace(); // TODO
        }
    }
    
    /**
     * Constructs a {@link ConstraintSyntaxTree} representing the given variable.
     * Negated if var starts with '-'.
     * 
     * @param var The string to turn into a {@link Variable}.
     * 
     * @return A {@link ConstraintSyntaxTree} representing the (negated) {@link Variable}.
     * @throws MalformedFileException If a malformed variable is found.
     */
    private ConstraintSyntaxTree getVariable(String var) throws MalformedFileException {
        // Check if variable is negated
        boolean negate = var.startsWith("-");
        if (negate) {
            negate = true;
            var = var.substring(1);
        }
        
        // Get variable number
        int variableNumber = 0;
        try {
            variableNumber = Integer.parseInt(var);
        } catch (NumberFormatException e) {
            throw new MalformedFileException("Invalid variable number: " + var);
        }
        
        // Construct CST
        ConstraintSyntaxTree variable = varPool.obtainVariable(variables.get(variableNumber));
        
        if (negate) {
            variable = new OCLFeatureCall(variable, OclKeyWords.NOT);
        }
        
        return variable;
    }
    
}
