package de.uni_hildesheim.de.see.model_extender.test;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import junit.framework.Assert;

import org.junit.Test;

import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.filter.ConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder;
import de.uni_hildesheim.sse.model.varModel.filter.FilterType;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder.VisibilityType;
import de.uni_hildesheim.sse.model_extender.convert.ModelExtender;
import de.uni_hildesheim.sse.model_extender.in.DimacsReader;
import de.uni_hildesheim.sse.model_extender.in.MalformedFileException;
import de.uni_hildesheim.sse.trans.in.ParserException;

/**
 * Class for testing the {@link ModelExtender}.
 * 
 * @author Adam Krafczyk
 */
public class ModelExtenderTest {

    /**
     * Tests whether constraints are added correctly to the model.
     */
    @Test
    public void testAddConstraints() {
        try {
            Project model = loadModel("testdata/testAddConstraint.dimacs");
            
            ModelExtender extender = new ModelExtender(model);
            
            DeclarationFinder declFinder = new DeclarationFinder(model, FilterType.ALL, null);
            Assert.assertEquals(0, declFinder.getVariableDeclarations(VisibilityType.ALL).size());
            
            ConstraintFinder constraintFinder = new ConstraintFinder(model);
            Assert.assertEquals(0, constraintFinder.getConstraints().size());
            
            extender.addConstraint("A || B");
            
            declFinder = new DeclarationFinder(model, FilterType.ALL, null);
            Assert.assertEquals(2, declFinder.getVariableDeclarations(VisibilityType.ALL).size());
            
            constraintFinder = new ConstraintFinder(model);
            Assert.assertEquals(1, constraintFinder.getConstraints().size());
            
        } catch (IOException | MalformedFileException | ParserException e) {
            e.printStackTrace();
            Assert.fail();
        }
    }
    
    /**
     * Tests whether the newly added constraints use the existing variables.
     */
    @Test
    public void testUseExistingVariables() {
        try {
            Project model = loadModel("testdata/testUseExistingVariables.dimacs");
            
            ModelExtender extender = new ModelExtender(model);
            
            DeclarationFinder declFinder = new DeclarationFinder(model, FilterType.ALL, null);
            Assert.assertEquals(2, declFinder.getVariableDeclarations(VisibilityType.ALL).size());
            
            ConstraintFinder constraintFinder = new ConstraintFinder(model);
            Assert.assertEquals(0, constraintFinder.getConstraints().size());
            
            extender.addConstraint("A || B || C");
            
            declFinder = new DeclarationFinder(model, FilterType.ALL, null);
            Assert.assertEquals(3, declFinder.getVariableDeclarations(VisibilityType.ALL).size());
            
            constraintFinder = new ConstraintFinder(model);
            Assert.assertEquals(1, constraintFinder.getConstraints().size());
            
        } catch (IOException | MalformedFileException | ParserException e) {
            e.printStackTrace();
            Assert.fail();
        }
    }
    
    /**
     * Loads the given model from a .dimacs file.
     * 
     * @param filename The filename of the model.
     * @return The model.
     * 
     * @throws MalformedFileException 
     * @throws IOException 
     * @throws FileNotFoundException 
     */
    private Project loadModel(String filename) throws FileNotFoundException, IOException, MalformedFileException {
        return new DimacsReader(new File(filename)).getModel();
    }
    
}
