package de.uni_hildesheim.sse.model_extender.convert;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.uni_hildesheim.sse.model.cst.CSTSemanticException;
import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.cst.VariablePool;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.BooleanType;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder.VisibilityType;
import de.uni_hildesheim.sse.model.varModel.filter.FilterType;
import de.uni_hildesheim.sse.trans.in.ParserException;
import de.uni_hildesheim.sse.trans.in.ParserException.ParserExceptionType;

/**
 * Extends existing models in CNF with additional constraints.
 * 
 * @author Adam Krafczyk
 */
public class ModelExtender {
    
    // VARIABLE_NAME
    // VARIABLE_NAME<=2
    private static final Pattern VARIABLE_PATTERN = Pattern.compile("^([\\w']+)([<>=]{0,2})(.*)$");

    private Project model;
    
    private Map<String, AbstractVariable> variables;
    
    private List<AbstractVariable> newVariables;
    
    private VariablePool varPool;
    
    /**
     * Creates a {@link ModelExtender} for the given model.
     * 
     * @param model The model where new constraints will be added to.
     */
    public ModelExtender(Project model) {
        this.model = model;
        this.varPool = new VariablePool();
        this.variables = new HashMap<String, AbstractVariable>();
        this.newVariables = new ArrayList<AbstractVariable>();
    }
    
    /**
     * Adds the given constraint to the model.
     * 
     * @param constraint The constraint to add to the model.
     * @throws ParserException If the constraint is not correctly formatted.
     */
    public void addConstraint(String constraint) throws ParserException {
        DeclarationFinder declarationFinder = new DeclarationFinder(model, FilterType.ALL, null);
        List<AbstractVariable> variableDeclarations = declarationFinder.getVariableDeclarations(VisibilityType.ALL);
        
        for (AbstractVariable var : variableDeclarations) {
            variables.put(var.getName(), var);
        }
        
        ConstraintSyntaxTree tree = toCST(constraint);
        Constraint tmp = new Constraint(null);
        try {
            tmp.setConsSyntax(tree);
        } catch (CSTSemanticException e) {
            e.printStackTrace(); // TODO
        }
        ListMaxTermConverter converter = new ListMaxTermConverter();
        converter.convert(tmp);
        for (ConstraintSyntaxTree cst : converter.getCSTs()) {
            addConstraint(cst);
        }
        
        for (AbstractVariable var : newVariables) {
            addAdditionalConstraints(var);
        }
        
        varPool.clear();
        variables.clear();
        newVariables.clear();
    }
    
    /**
     * Adds the given constraint to the model.
     * 
     * @param cst The {@link ConstraintSyntaxTree} to add to the model.
     */
    private void addConstraint(ConstraintSyntaxTree cst) {
        Constraint constraint = new Constraint(model);
        try {
            constraint.setConsSyntax(cst);
        } catch (CSTSemanticException e) {
            e.printStackTrace(); // TODO
        }
        model.add(constraint);
    }
    
    /**
     * Searches all variables with the given basename. All variables in the returned list
     * look like this: <code>&lt;basename&gt;[=,<,>,<=,>=]&lt;something&gt;</code>
     * 
     * @param baseName The beginning of the variable name.
     * @return A list of {@link AbstractVariable}s.
     */
    private List<AbstractVariable> findAllVaraiblesWithBaseName(String baseName) {
        List<AbstractVariable> list = new ArrayList<AbstractVariable>();
        
        for (AbstractVariable var : variables.values()) {
            String name = var.getName();
            if (name.startsWith(baseName) && name.length() > baseName.length()) {
                list.add(var);
            }
        }
        
        return list; 
    }
    
    /**
     * Adds additional needed constraints for new variables that contain <, > or =.
     * 
     * @param additionalVariable The variable that gets added to the model.
     */
    private void addAdditionalConstraints(AbstractVariable additionalVariable) {
        Matcher additionalVariableMatcher = VARIABLE_PATTERN.matcher(additionalVariable.getName());
        if (!additionalVariableMatcher.matches()) {
            System.err.println("Found invalid variable name: " + additionalVariable.getName());
        } else {
            List<AbstractVariable> variablesWithSameBasename
                = findAllVaraiblesWithBaseName(additionalVariableMatcher.group(1));
            
            
            String op1 = null;
            int val1 = 0;
            try {
                op1 = additionalVariableMatcher.group(2);
                val1 = Integer.parseInt(additionalVariableMatcher.group(3));
            } catch (NumberFormatException e) {
                op1 = null;
            }
            
            if (op1 == null) {
                System.err.println("Warning: adding completly new variable: " + additionalVariable.getName());
                return;
            }
            
            for (AbstractVariable variableWithSameBaseName : variablesWithSameBasename) {
                if (variableWithSameBaseName.equals(additionalVariable)) {
                    continue;
                }
                
                addAdditionalConstraints(additionalVariable, variableWithSameBaseName, op1, val1);
            }
        }
    }
    
    /**
     * Adds additional needed variables for the new variable that contains a >, < or = in combination
     * with the already existing variable that contains a >, < or =.
     * 
     * @param additionalVariable The variable that gets added to the model.
     * @param variableWithSameBaseName The already existing variable.
     * @param op1 The operator of additionalVariable, must be <, >, =, <= or >=.
     * @param val1 The value behind the operator of additionalVariable.
     */
    private void addAdditionalConstraints(AbstractVariable additionalVariable,
            AbstractVariable variableWithSameBaseName, String op1, int val1) {
        
        Matcher varMatcher = VARIABLE_PATTERN.matcher(variableWithSameBaseName.getName());
        
        if (!varMatcher.matches()) {
            System.err.println("Found invalid variable name: " + variableWithSameBaseName.getName());
            return;
        }
        
        Variable additional = varPool.obtainVariable(additionalVariable);
        Variable original = varPool.obtainVariable(variableWithSameBaseName);
        OCLFeatureCall notAdditional = new OCLFeatureCall(additional, OclKeyWords.NOT);
        OCLFeatureCall notOriginal = new OCLFeatureCall(original, OclKeyWords.NOT);
        try {
            String op2 = varMatcher.group(2);
            int val2 = Integer.parseInt(varMatcher.group(3));
            
            switch (intersect(op1, val1, op2, val2)) {
            case NONE:
                // additional -> !original AND original -> !additional
                // = additional XOR original
                // = (!additional OR !original) AND (additional OR original)
                addConstraint(new OCLFeatureCall(notAdditional, OclKeyWords.OR, notOriginal));
                addConstraint(new OCLFeatureCall(additional, OclKeyWords.OR, original));
                break;
                
            case ONE_IS_SUPERSET_OF_TWO:
                // additional is superset of original
                // original -> additional
                // = additional OR !original
                addConstraint(new OCLFeatureCall(additional, OclKeyWords.OR, notOriginal));
                break;
                
            case TWO_IS_SUPERSET_OF_ONE:
                // original is superset of additional
                // additional -> original
                // = original OR !additional
                addConstraint(new OCLFeatureCall(original, OclKeyWords.OR, notAdditional));
                break;
                
            case INTERSECT:
                // No constraints here
                break;
                
            case EQUAL:
                // additional = original
                // = additional -> original AND original -> additional
                // = original OR !additional AND additional OR !original
                addConstraint(new OCLFeatureCall(original, OclKeyWords.OR, notAdditional));
                addConstraint(new OCLFeatureCall(additional, OclKeyWords.OR, notOriginal));
                break;
                
            default:
                // Why does checkstyle want a default here?!    
            }
            
        } catch (NumberFormatException e) {
            // ignore, probably '=n' or so
        }
    }
    
    /**
     * Possible types of intersections.
     */
    private enum IntersectionType {
        /*
         * e.g. >2 and <1
         */
        NONE,
        /*
         * e.g. >2 and >3
         */
        ONE_IS_SUPERSET_OF_TWO,
        /*
         * e.g. <2 and <3
         */
        TWO_IS_SUPERSET_OF_ONE,
        /*
         * e.g. >2 and <3
         */
        INTERSECT,
        
        /*
         * e.g. =2 and =2 
         */
        EQUAL
    }
    
    /**
     * Checks whether 2 interval definitions (e.g. >2, <= 7) overlap.
     * 
     * @param op1 The first operator; must be "=", "<", ">", "<=" or ">=".
     * @param val1 The value for the first operator.
     * @param op2 The second operator; must be "=", "<", ">", "<=" or ">=".
     * @param val2 The value for the second operator.
     * @return The way the two intervals intersect.
     */
    private IntersectionType intersect(String op1, int val1, String op2, int val2) {
        IntersectionType result = null;
        
        if (op1.equals(">=")) {
            op1 = ">";
            val1 -= 1;
        } else if (op1.equals("<=")) {
            op1 = "<";
            val1 += 1;
        }
        
        if (op2.equals(">=")) {
            op2 = ">";
            val2 -= 1;
        } else if (op2.equals("<=")) {
            op2 = "<";
            val2 += 1;
        }
        
        if (op1.equals("=")) {
            result = handleOp1Equals(val1, op2, val2);
        } else if (op1.equals("<")) {
            result = handleOp1Lower(val1, op2, val2);
        } else if (op1.equals(">")) {
            result = handleOp1Greater(val1, op2, val2);
        } else {
            System.err.println("Unkown operator: " + op1);
        }
        
        return result;
    }

    /**
     * Handles the case in {@link ModelExtender#intersect(String, int, String, int)} when OP1 is >.
     * 
     * @param val1 The value for the first operator.
     * @param op2 The second operator; must be "=", "<", ">", "<=" or ">=".
     * @param val2 The value for the second operator.
     * @return The way the two intervals intersect.
     */
    private IntersectionType handleOp1Greater(int val1, String op2, int val2) {
        IntersectionType result = null;
        if (op2.equals("=")) {
            if (val1 < val2) {
                result = IntersectionType.ONE_IS_SUPERSET_OF_TWO;
            } else {
                result = IntersectionType.NONE;
            }
        } else if (op2.equals(">")) {
            if (val1 < val2) {
                result = IntersectionType.ONE_IS_SUPERSET_OF_TWO;
            } else if (val1 == val2) {
                result = IntersectionType.EQUAL;
            } else {
                result = IntersectionType.TWO_IS_SUPERSET_OF_ONE;
            }
        } else if (op2.equals("<")) {
            if (val1 + 1 <= val2 - 1) {
                result = IntersectionType.INTERSECT;
            } else {
                result = IntersectionType.NONE;
            }
        } else {
            System.err.println("Unkown operator: " + op2);
        }
        return result;
    }

    /**
     * Handles the case in {@link ModelExtender#intersect(String, int, String, int)} when OP1 is <.
     * 
     * @param val1 The value for the first operator.
     * @param op2 The second operator; must be "=", "<", ">", "<=" or ">=".
     * @param val2 The value for the second operator.
     * @return The way the two intervals intersect.
     */
    private IntersectionType handleOp1Lower(int val1, String op2, int val2) {
        IntersectionType result = null;
        if (op2.equals("=")) {
            if (val1 > val2) {
                result = IntersectionType.ONE_IS_SUPERSET_OF_TWO;
            } else {
                result = IntersectionType.NONE;
            }
        } else if (op2.equals("<")) {
            if (val1 > val2) {
                result = IntersectionType.ONE_IS_SUPERSET_OF_TWO;
            } else if (val1 == val2) {
                result = IntersectionType.EQUAL;
            } else {
                result = IntersectionType.TWO_IS_SUPERSET_OF_ONE;
            }
        } else if (op2.equals(">")) {
            if (val1 - 1 >= val2 + 1) {
                result = IntersectionType.INTERSECT;
            } else {
                result = IntersectionType.NONE;
            }
        } else {
            System.err.println("Unkown operator: " + op2);
        }
        return result;
    }

    /**
     * Handles the case in {@link ModelExtender#intersect(String, int, String, int)} when OP1 is =.
     * 
     * @param val1 The value for the first operator.
     * @param op2 The second operator; must be "=", "<", ">", "<=" or ">=".
     * @param val2 The value for the second operator.
     * @return The way the two intervals intersect.
     */
    private IntersectionType handleOp1Equals(int val1, String op2, int val2) {
        IntersectionType result = null;
        if (op2.equals("=")) {
            if (val1 == val2) {
                result = IntersectionType.EQUAL;
            } else {
                result = IntersectionType.NONE;
            }
        } else if (op2.equals(">")) {
            if (val1 > val2) {
                result = IntersectionType.TWO_IS_SUPERSET_OF_ONE;
            } else {
                result = IntersectionType.NONE;
            }
        } else if (op2.equals("<")) {
            if (val1 < val2) {
                result = IntersectionType.TWO_IS_SUPERSET_OF_ONE;
            } else {
                result = IntersectionType.NONE;
            }
        } else {
            System.err.println("Unkown operator: " + op2);
        }
        return result;
    }
    
    /**
     * Converts the given String into a {@link ConstraintSyntaxTree}.
     * 
     * @param constraint The String that contains the constraint.
     * @return A {@link ConstraintSyntaxTree} representing the constraint.
     * @throws ParserException If the constraint is not correctly formatted.
     */
    private ConstraintSyntaxTree toCST(String constraint) throws ParserException {
        constraint = constraint.replace(" ", "");
        
        // We assume that ! can only be in front of variables
        
        String highestOperand = null;
        int highestOperandLevel = Integer.MAX_VALUE;
        int highestOperandPos = -1;
        int currentLevel = 0;
        
        for (int i = 0; i < constraint.length(); i++) {
            char c = constraint.charAt(i);
            switch (c) {
            case '(':
                currentLevel++;
                break;
            case ')':
                currentLevel--;
                break;
                
            case '&':
                if (constraint.charAt(i + 1) != '&') {
                    throw new ParserException(ParserExceptionType.NOT_SUPPORTED_FIELD);
                }
                if (highestOperandLevel > currentLevel) {
                    highestOperand = OclKeyWords.AND;
                    highestOperandPos = i;
                    highestOperandLevel = currentLevel;
                }
                i++;
                break;
            
            case '|':
                if (constraint.charAt(i + 1) != '|') {
                    throw new ParserException(ParserExceptionType.NOT_SUPPORTED_FIELD);
                }
                if (highestOperandLevel > currentLevel) {
                    highestOperand = OclKeyWords.OR;
                    highestOperandPos = i;
                    highestOperandLevel = currentLevel;
                }
                i++;
                break;
            
            default:
                break;
            }
        }
        
        ConstraintSyntaxTree result = null;
        
        if (highestOperandPos == -1) {
            if (!constraint.equals("y")) {
                // we only have a variable
                result = getVariable(constraint);
            }
            // result = null if condition is "y"
        } else {
            StringBuffer left = new StringBuffer();
            StringBuffer right = new StringBuffer();
            split(constraint, highestOperandPos, highestOperandLevel, left, right);
            result = new OCLFeatureCall(toCST(left.toString()),
                    highestOperand, toCST(right.toString()));
        }
        
        return result;
    }
    
    /**
     * Splits the given line at the given operand position into a left and right part.
     * @param line The whole line
     * @param operandPos The position of the operand in the line
     * @param operandLevel The logic level of the operand (by counting brackets)
     * @param left A {@link StringBuffer} where the left side of the operand will be written to
     * @param right A {@link StringBuffer} where the right side of the operand will be written to
     */
    private void split(String line, int operandPos, int operandLevel, StringBuffer left, StringBuffer right) {
        left.append(line.substring(0, operandPos));
        right.append(line.substring(operandPos + 2, line.length()));
        int currentLevel = 0;
        for (int i = 0; i < left.length(); i++) {
            char c = left.charAt(i);
            if ((c == '(' || c == ')') && currentLevel < operandLevel) {
                left.replace(i, i + 1, "");
            }
            if (c == '(') {
                currentLevel++;
            }
            if (c == ')') {
                currentLevel--;
            }
        }
        currentLevel = 0;
        for (int i = right.length() - 1; i >= 0; i--) {
            char c = right.charAt(i);
            if ((c == '(' || c == ')') && currentLevel < operandLevel) {
                right.replace(i, i + 1, "");
            }
            if (c == ')') {
                currentLevel++;
            }
            if (c == '(') {
                currentLevel--;
            }
        }
    }
    
    /**
     * Converts the given variable from a string to a {@link ConstraintSyntaxTree}.
     * A ! at the beginning will result in a {@link OCLFeatureCall} with NOT.
     * @param var A string representing the variable.
     * @return A {@link ConstraintSyntaxTree} representing the variable.
     */
    private ConstraintSyntaxTree getVariable(String var) {
        ConstraintSyntaxTree result = null;
        
        // Remove parenthesis around variable names
        if (var.startsWith("(")) {
            var = var.substring(1);
            if (var.endsWith(")")) {
                var = var.substring(0, var.length() - 1);
            }
        }
        
        if (var.startsWith("!")) {
            var = var.substring(1);
            ConstraintSyntaxTree variable = getVariable(var);
            result = new OCLFeatureCall(variable, OclKeyWords.NOT);
        }
        
//        TODO: is this needed?
//        if (result == null) {
//            result = handleConstantInVariable(var);
//        }
//        
//        if (result == null) {
//            result = handleModuleInVariable(var);
//        }
//        
//        if (result == null) {
//            result = handleStringInVariable(var);
//        }
        
        if (result == null) {
            // Remove ' around variable names
            if (var.startsWith("'")) {
                var = var.substring(1);
            }
            if (var.endsWith("'")) {
                var = var.substring(0, var.length() - 1);
            }
            
            Variable variable = varPool.obtainVariable(getVariableFromProject(var));
            result = variable;
        }
        return result;
    }
    
    /**
     * Gets or creates the variable for the given name in the model.
     * 
     * @param name The name of the variable.
     * @return The variable.
     */
    private AbstractVariable getVariableFromProject(String name) {
        AbstractVariable var = variables.get(name);
        if (var == null) {
            var = new DecisionVariableDeclaration(name, BooleanType.TYPE, model);
            model.add(var);
            variables.put(name, var);
            newVariables.add(var);
        }
        return var;
    }
    
}
