package de.uni_hildesheim.sse.trans.convert;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import de.uni_hildesheim.sse.model.cst.CSTSemanticException;
import de.uni_hildesheim.sse.model.cst.Comment;
import de.uni_hildesheim.sse.model.cst.CompoundAccess;
import de.uni_hildesheim.sse.model.cst.CompoundInitializer;
import de.uni_hildesheim.sse.model.cst.ConstantValue;
import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.ContainerInitializer;
import de.uni_hildesheim.sse.model.cst.ContainerOperationCall;
import de.uni_hildesheim.sse.model.cst.IConstraintTreeVisitor;
import de.uni_hildesheim.sse.model.cst.IfThen;
import de.uni_hildesheim.sse.model.cst.Let;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Parenthesis;
import de.uni_hildesheim.sse.model.cst.Self;
import de.uni_hildesheim.sse.model.cst.UnresolvedExpression;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;

/**
 * Searches variable declartions in a CST sorted by the "depth" in the tree.
 * 
 * @author Adam Krafczyk
 */
public class DeclarationInConstraintFinderWithDepth implements IConstraintTreeVisitor {

    private Map<AbstractVariable, Integer> variables;
    
    private int currentDepth;
    
    /**
     * Sole constructor for this class.
     * @param cst A constraint where all nested {@link AbstractVariable}'s should be found.
     */
    public DeclarationInConstraintFinderWithDepth(ConstraintSyntaxTree cst) {
        variables = new HashMap<AbstractVariable, Integer>();
        currentDepth = 0;
        cst.accept(this);
    }
    
    /**
     * Getter for a sorted {@link List} of {@link AbstractVariable}s.
     * 
     * @return A {@link List} of variable declarations sorted by the depth in the CST.
     */
    public List<AbstractVariable> getDeclarationsInOrder() {
        Set<Map.Entry<AbstractVariable, Integer>> entrySet = variables.entrySet();
        @SuppressWarnings("unchecked")
        Map.Entry<AbstractVariable, Integer>[] entryArray = new Map.Entry[entrySet.size()];
        entryArray = entrySet.toArray(entryArray);
        
        Arrays.sort(entryArray, new Comparator< Map.Entry<AbstractVariable, Integer>>() {
            @Override
            public int compare(Map.Entry<AbstractVariable, Integer> o1, Map.Entry<AbstractVariable, Integer> o2) {
                int result = 0;
                
                if (o1.getValue() < o2.getValue()) {
                    result = -1;
                } else if (o1.getValue() > o2.getValue()) {
                    result = 1;
                }
                
                return result;
            }
        });
        
        ArrayList<AbstractVariable> list = new ArrayList<AbstractVariable>();
        for (Map.Entry<AbstractVariable, Integer> entry : entryArray) {
            list.add(entry.getKey());
        }
        return list;
    }
    
    /**
     * Adds the variable with the current depth or updates the depth if its smaller.
     * 
     * @param var The variable to add.
     */
    private void addVariable(AbstractVariable var) {
        if (!variables.containsKey(var) || (variables.get(var) > currentDepth)) {
            variables.put(var, currentDepth);
        }
    }
    
    @Override
    public void visitConstantValue(ConstantValue value) {
        // No function needed
    }

    @Override
    public void visitVariable(Variable variable) {
        addVariable(variable.getVariable());
    }

    @Override
    public void visitParenthesis(Parenthesis parenthesis) {
        parenthesis.getExpr().accept(this);
    }

    @Override
    public void visitComment(Comment comment) {
        comment.getExpr().accept(this);
    }

    @Override
    public void visitOclFeatureCall(OCLFeatureCall call) {
        if (!call.getOperation().equals(OclKeyWords.NOT)) { // A not doesn't count
            currentDepth++;
        }
        if (null != call.getOperand()) { // user defined function!
            call.getOperand().accept(this);
        }
        for (int i = 0; i < call.getParameterCount(); i++) {
            call.getParameter(i).accept(this);
        }
        if (!call.getOperation().equals(OclKeyWords.NOT)) {
            currentDepth--;
        }
    }

    @Override
    public void visitLet(Let let) {
        addVariable(let.getVariable());
        currentDepth++;
        let.getInExpression().accept(this);
        currentDepth--;  
    }

    @Override
    public void visitIfThen(IfThen ifThen) {
        currentDepth++;
        ifThen.getIfExpr().accept(this);
        ifThen.getThenExpr().accept(this);
        ifThen.getElseExpr().accept(this);
        currentDepth--;
    }

    @Override
    public void visitContainerOperationCall(ContainerOperationCall call) {
        currentDepth++;
        call.getContainer().accept(this);
        call.getExpression().accept(this);
        for (int i = 0; i < call.getDeclaratorsCount(); i++) {
            addVariable(call.getDeclarator(i));
        } 
        currentDepth--;
    }

    @Override
    public void visitCompoundAccess(CompoundAccess access) {
        // access.inferDatatype() must be called before access.getResolvedSlot() can be called 
        try {
            access.inferDatatype();
        } catch (CSTSemanticException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        addVariable(access.getResolvedSlot());
        currentDepth++;
        access.getCompoundExpression().accept(this);
        currentDepth--;
    }

    @Override
    public void visitUnresolvedExpression(UnresolvedExpression expression) {
        // TODO Auto-generated method stub
    }

    @Override
    public void visitCompoundInitializer(CompoundInitializer initializer) {
        // TODO Auto-generated method stub
    }

    @Override
    public void visitContainerInitializer(ContainerInitializer initializer) {
        // TODO Auto-generated method stub
    }

    @Override
    public void visitSelf(Self self) {
        // no variable declaration
    }
    

}
