package de.uni_hildesheim.sse.trans.convert;

import java.io.PrintStream;
import java.io.StringWriter;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.io.output.WriterOutputStream;

import de.uni_hildesheim.sse.model.confModel.AssignmentState;
import de.uni_hildesheim.sse.model.confModel.Configuration;
import de.uni_hildesheim.sse.model.confModel.ConfigurationException;
import de.uni_hildesheim.sse.model.confModel.IDecisionVariable;
import de.uni_hildesheim.sse.model.cst.CSTSemanticException;
import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.DebugConstraintTreeVisitor;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Parenthesis;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.cstEvaluation.EvaluationVisitor;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.model.varModel.filter.DeclrationInConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.values.BooleanValue;
import de.uni_hildesheim.sse.trans.Main;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Converts a boolean formula in a list of disjunction terms by means of
 * <a href="http://de.wikipedia.org/w/index.php?title=Konjunktive_Normalform&oldid=130361119
 * #Beispiel_f.C3.BCr_die_Bildung">max terms</a>.
 * @author Adam Krafczyk
 * @author El-Sharkawy
 *
 */
public class MaxTermConverter implements ICNFConvertStrategy {
    public static final BigInteger TWO = BigInteger.valueOf(2);
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(MaxTermConverter.class, Main.ID);

    private static final int MIN_SIMPLYIFY_LENGTH = 18;
    
    /**
     * Should avoid that complex constraints are translated twice.
     */
    private Set<ConstraintSyntaxTree> complexConstraints = new HashSet<ConstraintSyntaxTree>();
    private Project project;
    private Map<AbstractVariable, Variable> variablesCache = new HashMap<AbstractVariable, Variable>();
    
    /**
     * Returns a {@link Variable} for the given {@link AbstractVariable}.
     * This method will cache already created {@link Variable}s to reduce memory consumption.
     * @param decl A {@link AbstractVariable} for which a {@link Variable} shall be created.
     * @return A cached {@link Variable} for the given {@link AbstractVariable} or a new one
     *     if no cached variable exists, a new one is created and will be cached for future calls.
     */
    protected final Variable getVariable(AbstractVariable decl) {
        Variable variable = variablesCache.get(decl);
        if (null == variable) {
            variable = new Variable(decl);
            variablesCache.put(decl, variable);
        }
        
        return variable;
    }
    
    @Override
    public void convert(Constraint constraint) {
        project = (Project) constraint.getTopLevelParent();
        
        createCNFParts(constraint.getConsSyntax());
    }
    
    /**
     * Returns the "length" of a {@link ConstraintSyntaxTree}.
     * @param cst The {@link ConstraintSyntaxTree} to get the "length" of.
     * @return The "length" (i.e. the number of variables in the {@link ConstraintSyntaxTree}).
     */
    private int getSyntaxTreeLength(ConstraintSyntaxTree cst) {
        return new DeclrationInConstraintFinder(cst).getDeclarations().size();
    }
    
    /**
     * Converts the given expression into CNF and adds it via
     * {@link MaxTermConverter#addConstraint(ConstraintSyntaxTree)}.
     * @param originalConstraint A boolean expression
     */
    private void createCNFParts(ConstraintSyntaxTree originalConstraint) {
        boolean handled = false;
        
        // Unpack constraint if there are parenthesis at the top level
        if (originalConstraint instanceof Parenthesis) {
            createCNFParts(((Parenthesis) originalConstraint).getExpr());
            handled = true;
        }
        
        if (!handled && getSyntaxTreeLength(originalConstraint) >= MIN_SIMPLYIFY_LENGTH) {
            originalConstraint = simplifyComplexConstraint(originalConstraint);
        }
        
        // Split top level AND calls into two separate terms
        if (!handled && originalConstraint instanceof OCLFeatureCall) {
            OCLFeatureCall call = (OCLFeatureCall) originalConstraint;
            if (call.getOperation().equals(OclKeyWords.AND)) {
                LOGGER.info("Highest operation is an AND, splitting");
                createCNFParts(call.getOperand());
                createCNFParts(call.getParameter(0));
                handled = true;
            }
        }
        
        // Stop recursion and start translation!
        if (!handled) {
            if (!complexConstraints.contains(originalConstraint)) {
                complexConstraints.add(originalConstraint);
                
                // Get an array of all variables in the constraint
                DeclarationInConstraintFinderWithDepth finder =
                    new DeclarationInConstraintFinderWithDepth(originalConstraint);
                AbstractVariable[] declarationArray = finder.getDeclarationsInOrder()
                    .toArray(new AbstractVariable[] {});        
                StringWriter sWriter = new StringWriter();
                WriterOutputStream outStream = new WriterOutputStream(sWriter);
                originalConstraint.accept(new DebugConstraintTreeVisitor(new PrintStream(outStream)));
                String parsedConstraint = sWriter.toString();
                if (null != parsedConstraint && !parsedConstraint.isEmpty()) {
                    LOGGER.info(parsedConstraint);
                }
                
                for (AbstractVariable var : declarationArray) {
                    LOGGER.debug(var.getName());
                }
                
                // Create a project which only contains our single Constraint
                Project singleConstraintProject = new Project("SingleConstraintProject");
                Constraint constraintCopy = null;
                try {
                    constraintCopy = new Constraint(originalConstraint, singleConstraintProject);
                } catch (CSTSemanticException e) {
                    // Cannot happen
                    LOGGER.exception(e);
                }
                singleConstraintProject.add(constraintCopy);
                for (AbstractVariable var : declarationArray) {
                    singleConstraintProject.add(var);
                }
                
                // Create a configuration object for the singleConstraintProject
                Configuration config = new Configuration(singleConstraintProject);
                
                if (declarationArray.length >= 16) {
                    handleBigConstraint(originalConstraint, declarationArray, config);
                } else {
                    handleSmallConstraint(originalConstraint, declarationArray, config);
                }
            } else {
                LOGGER.info("Nothing to do, constraint was already part of another translation step.");
            }
        }
    }

    /**
     * Tries to rewrite/split constraints with more than {@value #MIN_SIMPLYIFY_LENGTH} variables to constraints
     * with less variables.
     * @param originalConstraint A constraint with at least {@value #MIN_SIMPLYIFY_LENGTH} variables.
     * @return A constraint which maybe be more in CNF form as before, maybe the same constraint as before.
     */
    private ConstraintSyntaxTree simplifyComplexConstraint(ConstraintSyntaxTree originalConstraint) {
        LOGGER.info("Constraint is longer than " + MIN_SIMPLYIFY_LENGTH + ", trying to simplify");
        
        // pull in NOT's as far as possible
        CSTNegater negater = new CSTNegater();
        originalConstraint.accept(negater);
        originalConstraint = negater.getResult();
        
        // expand to get closer to CNF
        CSTExpander expander = new CSTExpander(MIN_SIMPLYIFY_LENGTH);
        int numExpanded = 0;
        do {
            expander.clearResult();
            originalConstraint.accept(expander);
            if (expander.getResult() != null) {
                originalConstraint = expander.getResult();
                numExpanded++;
            }
        } while (expander.getResult() != null);
        
        LOGGER.info("Found " + numExpanded + " possible expansions to simplify constraint");
        return originalConstraint;
    }

    /**
     * Converts a small constraint into a CNF formula.
     * This constraint must have at maximum 15 different {@link AbstractVariable}s.
     * @param originalConstraint The constraint which shall be converted into CNF formula.
     *     this constraint may consists only of AND, OR, and NOT operations.
     * @param declarationArray The {@link AbstractVariable}s, which are used inside the <tt>originalConstraint</tt>.
     * @param config A temporary {@link Configuration} containing only <tt>originalConstraint</tt>
     *     and <tt>declarationArray</tt>
     */
    protected void handleSmallConstraint(ConstraintSyntaxTree originalConstraint, AbstractVariable[] declarationArray,
        Configuration config) {
        
        EvaluationVisitor evalVisitor = new EvaluationVisitor();
        evalVisitor.init(config, null, false, null);
        // for each combination of the variables (true or false)
        for (int i = 0; i < Math.pow(2, declarationArray.length); i++) {
            boolean[] state = new boolean[declarationArray.length];
            for (int j = 0; j < state.length; j++) {
                IDecisionVariable decisionVar = config.getDecision(declarationArray[j]);
                /*
                 * i is a bit-field where each bit represents one variable
                 * j is the current variable
                 * To test if variable #j is true check if the bit #j in i is 0 or 1
                 */
                if ((i & (1 << j)) != 0) {
                    state[j] = true;
                    try {
                        decisionVar.setValue(BooleanValue.TRUE, AssignmentState.ASSIGNED);
                    } catch (ConfigurationException e) {
                        // TODO
                        LOGGER.exception(e);
                    }
                } else {
                    state[j] = false;
                    try {
                        decisionVar.setValue(BooleanValue.FALSE, AssignmentState.ASSIGNED);
                    } catch (ConfigurationException e) {
                        // TODO
                        LOGGER.exception(e);
                    }
                }
            }
            
            // get the result
            originalConstraint.accept(evalVisitor);
            boolean result = evalVisitor.constraintFulfilled();
            
            // if the result it false, add the negated combination to the list of expressions
            if (!result) {
                addConstraint(createNegatedORExpression(declarationArray, state)); 
            }
            
            // 1 * init() + n * clearResult() is much faster than n * (init() + clear())
            evalVisitor.clearResult();
        }
    }
    
    /**
     * Converts a constraint into a CNF formula.
     * This constraint can have any number of {@link AbstractVariable}s, but for small constraints
     * {@link MaxTermConverter#handleSmallConstraint} is recommended.
     * @param originalConstraint The constraint which shall be converted into CNF formula.
     *     this constraint may consists only of AND, OR, and NOT operations.
     * @param declarationArray The {@link AbstractVariable}s, which are used inside the <tt>originalConstraint</tt>.
     * @param config A temporary {@link Configuration} containing only <tt>originalConstraint</tt>
     *     and <tt>declarationArray</tt>
     */
    protected void handleBigConstraint(ConstraintSyntaxTree originalConstraint, AbstractVariable[] declarationArray,
            Configuration config) {
        
        EvaluationVisitor evalVisitor = new EvaluationVisitor();
        evalVisitor.init(config, null, false, null);
        // for each combination of the variables (true or false)
        for (BigInteger i = BigInteger.ZERO; i.compareTo(TWO.pow(declarationArray.length)) == -1;
            i = i.add(BigInteger.ONE)) {
            
            boolean[] state = new boolean[declarationArray.length];
            for (int j = 0; j < state.length; j++) {
                IDecisionVariable decisionVar = config.getDecision(declarationArray[j]);
                /*
                 * i is a bit-field where each bit represents one variable
                 * j is the current variable
                 * To test if variable #j is true check if the bit #j in i is 0 or 1
                 */
                if (i.testBit(j)) {
                    state[j] = true;
                    try {
                        decisionVar.setValue(BooleanValue.TRUE, AssignmentState.ASSIGNED);
                    } catch (ConfigurationException e) {
                        // TODO
                        LOGGER.exception(e);
                    }
                } else {
                    state[j] = false;
                    try {
                        decisionVar.setValue(BooleanValue.FALSE, AssignmentState.ASSIGNED);
                    } catch (ConfigurationException e) {
                        // TODO
                        LOGGER.exception(e);
                    }
                }
            }
            
            // get the result
            originalConstraint.accept(evalVisitor);
            boolean result = evalVisitor.constraintFulfilled();
            
            // if the result it false, add the negated combination to the list of expressions
            if (!result) {
                addConstraint(createNegatedORExpression(declarationArray, state)); 
            }
            
            // 1 * init() + n * clearResult() is much faster than n * (init() + clear())
            evalVisitor.clearResult();
        }
    }
    
    /**
     * Creates an {@link ConstraintSyntaxTree} with the variables negated and OR'd together.
     * @param variables Array of Variables
     * @param states Array whether each variable is true or false
     * @return the {@link ConstraintSyntaxTree}
     */
    protected ConstraintSyntaxTree createNegatedORExpression(AbstractVariable[] variables, boolean[] states) {
        ConstraintSyntaxTree call = null;
        if (!states[0]) {
            call = getVariable(variables[0]);
        } else {
            call = new OCLFeatureCall(getVariable(variables[0]), OclKeyWords.NOT);
        }
        
        for (int i = 1; i < states.length; i++) {
            ConstraintSyntaxTree variable = getVariable(variables[i]);
            if (states[i]) {
                variable = new OCLFeatureCall(variable, OclKeyWords.NOT);
            }
            call = new OCLFeatureCall(call, OclKeyWords.OR, variable);
        }
        return call;
    }

    /**
     * Adds the given constraint to the project.
     * @param cst The constraint to add.
     */
    protected void addConstraint(ConstraintSyntaxTree cst) {
        Constraint constraint = new Constraint(project);
        try {
            constraint.setConsSyntax(cst);
        } catch (CSTSemanticException e) {
            LOGGER.exception(e);
        }
        
        project.add(constraint);
    }
}
