package de.uni_hildesheim.see.model_extender;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.StringWriter;

import org.junit.Assert;
import org.junit.Test;

import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.filter.ConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder.VisibilityType;
import de.uni_hildesheim.sse.model.varModel.filter.FilterType;
import de.uni_hildesheim.sse.model_extender.convert.ModelExtender;
import de.uni_hildesheim.sse.model_extender.in.DimacsReader;
import de.uni_hildesheim.sse.model_extender.in.MalformedFileException;
import de.uni_hildesheim.sse.trans.DimacsTestUtils;
import de.uni_hildesheim.sse.trans.in.ParserException;
import de.uni_hildesheim.sse.trans.out.DimacsWriter;

/**
 * Class for testing the {@link ModelExtender}.
 * 
 * @author Adam Krafczyk
 */
public class ModelExtenderTest {

    /**
     * Tests whether constraints are added correctly to the model.
     */
    @Test
    public void testAddConstraints() {
        try {
            Project model = loadModel("testdata/testAddConstraint.dimacs");
            
            ModelExtender extender = new ModelExtender(model);
            
            DeclarationFinder declFinder = new DeclarationFinder(model, FilterType.ALL, null);
            Assert.assertEquals(0, declFinder.getVariableDeclarations(VisibilityType.ALL).size());
            
            ConstraintFinder constraintFinder = new ConstraintFinder(model);
            Assert.assertEquals(0, constraintFinder.getConstraints().size());
            
            extender.addConstraint("A || B");
            
            declFinder = new DeclarationFinder(model, FilterType.ALL, null);
            Assert.assertEquals(2, declFinder.getVariableDeclarations(VisibilityType.ALL).size());
            
            constraintFinder = new ConstraintFinder(model);
            Assert.assertEquals(1, constraintFinder.getConstraints().size());
            
        } catch (IOException | MalformedFileException | ParserException e) {
            e.printStackTrace();
            Assert.fail();
        }
    }
    
    /**
     * Tests whether the newly added constraints use the existing variables.
     */
    @Test
    public void testUseExistingVariables() {
        try {
            Project model = loadModel("testdata/testUseExistingVariables.dimacs");
            
            ModelExtender extender = new ModelExtender(model);
            
            DeclarationFinder declFinder = new DeclarationFinder(model, FilterType.ALL, null);
            Assert.assertEquals(2, declFinder.getVariableDeclarations(VisibilityType.ALL).size());
            
            ConstraintFinder constraintFinder = new ConstraintFinder(model);
            Assert.assertEquals(0, constraintFinder.getConstraints().size());
            
            extender.addConstraint("A || B || C");
            
            declFinder = new DeclarationFinder(model, FilterType.ALL, null);
            Assert.assertEquals(3, declFinder.getVariableDeclarations(VisibilityType.ALL).size());
            
            constraintFinder = new ConstraintFinder(model);
            Assert.assertEquals(1, constraintFinder.getConstraints().size());
            
        } catch (IOException | MalformedFileException | ParserException e) {
            e.printStackTrace();
            Assert.fail();
        }
    }
    
    /**
     * Tests whether additional constraints for variables with <, > or = are
     * added correctly.
     */
    @Test
    public void testAdditionalConstraints() {
        try {
            Project model = loadModel("testdata/testAdditionalConstraints.dimacs");
            
            ModelExtender extender = new ModelExtender(model);
            
            extender.addConstraint("A<6");
            extender.addConstraint("A<4");
            
            extender.addConstraint("B>1");
            extender.addConstraint("B>2");
            
            extender.addConstraint("C=8");
            
            extender.addConstraint("D>=2");
            extender.addConstraint("D>=6");
            extender.addConstraint("D>1");
            
            
            extender.addConstraint("E<=8");
            extender.addConstraint("E<=3");
            extender.addConstraint("E<4");
            
            StringWriter dimacsStringWriter = new StringWriter();
            DimacsWriter writer = new DimacsWriter(model, dimacsStringWriter);
            writer.write();
            
            String dimacsString = dimacsStringWriter.toString();
            
            testA(dimacsString);
            testB(dimacsString);
            testC(dimacsString);
            testD(dimacsString);
            testE(dimacsString);
            
        } catch (IOException | MalformedFileException | ParserException e) {
            e.printStackTrace();
            Assert.fail();
        }
    }

    /**
     * Tests whether the correct constraints for the variable E are added.
     * 
     * @param dimacsString The DIMACS model as string.
     */
    private void testE(String dimacsString) {
        int eE8 = DimacsTestUtils.getNumberOfVariable(dimacsString, "E=8");
        int eLE8 = DimacsTestUtils.getNumberOfVariable(dimacsString, "E<=8");
        int eLE3 = DimacsTestUtils.getNumberOfVariable(dimacsString, "E<=3");
        int eL4 = DimacsTestUtils.getNumberOfVariable(dimacsString, "E<4");
        
        // E<=8 is superset of E=8
        //  E=8 -> E<=8 == E<=8 || !E=8
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, eLE8, -eE8));
        
        // E=8 and E<=3 do not intersect:
        //  E=8 XOR E<=3 == !E=8 || !E<=3 AND E=8 || E<=3
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -eE8, -eLE3));
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, eE8, eLE3));
        
        // E<=8 is superset of E<=3
        //  E<=3 -> E<=8 == E<=8 || !E<=3
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, eLE8, -eLE3));
        
        // E<4 is equal to E<=3
        //  E<4 = E<=3 == !E<=3 || E<4 AND E<=3 || !E<4
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -eLE3, eL4));
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, eLE3, -eL4));
    }

    /**
     * Tests whether the correct constraints for the variable D are added.
     * 
     * @param dimacsString The DIMACS model as string.
     */
    private void testD(String dimacsString) {
        int dE5 = DimacsTestUtils.getNumberOfVariable(dimacsString, "D=5");
        int dGE2 = DimacsTestUtils.getNumberOfVariable(dimacsString, "D>=2");
        int dGE6 = DimacsTestUtils.getNumberOfVariable(dimacsString, "D>=6");
        int dG1 = DimacsTestUtils.getNumberOfVariable(dimacsString, "D>1");
        
        // D>=2 is superset of D=5
        //  D=5 -> D>=2 == D>=2 || !D=5
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, dGE2, -dE5));
        
        // D=5 and D>=6 do not intersect:
        //  D=5 XOR D>=6 == !D=5 || !D>=6 AND D=5 || D>=6
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -dE5, -dGE6));
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, dE5, dGE6));
        
        // D>=2 is superset of D>=6
        //  D>=6 -> D>=2 == D>=2 || !D>=6
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, dGE2, -dGE6));
        
        // D>1 is equal to D>=2
        //  D>1 = D>=2 == !D>=2 || D>1 AND D>=2 || !D>1
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -dGE2, dG1));
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, dGE2, -dG1));
    }

    /**
     * Tests whether the correct constraints for the variable C are added.
     * 
     * @param dimacsString The DIMACS model as string.
     */
    private void testC(String dimacsString) {
        int cE9 = DimacsTestUtils.getNumberOfVariable(dimacsString, "C=9");
        int cE8 = DimacsTestUtils.getNumberOfVariable(dimacsString, "C=8");
        
        // C=9 and C=9 do not intersect:
        //  C=9 XOR C=8 == !C=9 || !C=8 AND C=9 || C=8
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -cE8, -cE9));
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, cE8, cE9));
    }

    /**
     * Tests whether the correct constraints for the variable B are added.
     * 
     * @param dimacsString The DIMACS model as string.
     */
    private void testB(String dimacsString) {
        int bE2 = DimacsTestUtils.getNumberOfVariable(dimacsString, "B=2");
        int bG2 = DimacsTestUtils.getNumberOfVariable(dimacsString, "B>2");
        int bG1 = DimacsTestUtils.getNumberOfVariable(dimacsString, "B>1");
        
        // B>1 is superset of B=2:
        //  B=2 -> B>1 == B>1 || !B=2
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -bE2, bG1));
        
        // B=2 and B>2 do not intersect:
        //  B=2 XOR B>2 == !B=2 || !B>2 AND B=2 || B>2
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -bE2, -bG2));
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, bE2, bG2));
        
        // B>1 is superset of B>2:
        //  B>2 -> B>1 == B>1 || !B>2
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -bG2, bG1));
    }

    /**
     * Tests whether the correct constraints for the variable A are added.
     * 
     * @param dimacsString The DIMACS model as string.
     */
    private void testA(String dimacsString) {
        int aE5 = DimacsTestUtils.getNumberOfVariable(dimacsString, "A=5");
        int aL6 = DimacsTestUtils.getNumberOfVariable(dimacsString, "A<6");
        int aL4 = DimacsTestUtils.getNumberOfVariable(dimacsString, "A<4");
        
        // A<6 is superset of A=5:
        //  A=5 -> A<6 == A<6 || !A=5
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -aE5, aL6));
        
        // A=6 and A<4 do not intersect:
        //  A=5 XOR A<4 == !A=5 || !A<4 AND A=5 || A<4
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -aE5, -aL4));
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, aE5, aL4));
        
        // A<6 is superset of A<4:
        //  A<4 -> A<6 == A<6 || !A<4
        Assert.assertTrue(DimacsTestUtils.containsConstraint(dimacsString, -aL4, aL6));
    }
    
    /**
     * Loads the given model from a .dimacs file.
     * 
     * @param filename The filename of the model.
     * @return The model.
     * 
     * @throws MalformedFileException 
     * @throws IOException 
     * @throws FileNotFoundException 
     */
    private Project loadModel(String filename) throws FileNotFoundException, IOException, MalformedFileException {
        return new DimacsReader(new File(filename)).getModel();
    }
    
}
