package de.uni_hildesheim.sse.trans.in.rsf;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.Main;
import de.uni_hildesheim.sse.trans.in.ParserException;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Contains a Choice and the ChoiceItems.
 * @see <a href="https://projects.sse.uni-hildesheim.de/agilo/ModelTranslator/wiki/
 *Specification/RSF2DIMACS#TranslationofChoices">Specifikation</a>
 * 
 * @author Adam Krafczyk
 */
class RSFChoice extends RSFCondition {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(RSFChoice.class, Main.ID);
    
    private RSFItem choiceItem;
    
    /**
     * Stores all choice items and their depends conditions.
     */
    private Map<RSFItem, List<String>> choiceItems;
    
    private String choiceCondition;
    
    /**
     * Creates a new Choice with no ChoiceItems.
     * @param choiceItem The name of the Choice.
     */
    RSFChoice(RSFItem choiceItem) {
        this.choiceItem = choiceItem;
        choiceItems = new HashMap<RSFItem, List<String>>();
    }
    
    /**
     * Adds a ChoiceItem to the Choice.
     * @param item The name of the ChoiceItem.
     */
    void addChoiceItem(RSFItem item) {
        if (!choiceItems.containsKey(item)) {
            choiceItems.put(item, new ArrayList<String>());
        }
    }
    
    /**
     * Adds a constraint of a choice item of this choice.
     * 
     * @param item The choice item.
     * @param condition The condition as read from a depends line in the .rsf file.
     */
    void addChoiceItemDepends(RSFItem item, String condition) {
        // TODO error handling if only one is true?
        if (condition.startsWith("\"") && condition.endsWith("\"")) {
            condition = condition.substring(1, condition.length() - 1);
        }
        
        List<String> constraints = choiceItems.get(choiceItem);
        if (constraints != null) {
            constraints.add(condition);
        } else {
            constraints = new ArrayList<String>();
            constraints.add(condition);
            choiceItems.put(item, constraints);
        }
    }
    
    /**
     * Adds a constraint the choice.
     * @param condition A condition as read from a depends line in the .rsf file.
     */
    void addChoiceCondition(String condition) {
        // TODO error handling if only one is true?
        if (condition.startsWith("\"") && condition.endsWith("\"")) {
            condition = condition.substring(1, condition.length() - 1);
        }
        
        if (choiceCondition == null) {
            choiceCondition = condition;
        } else {
            choiceCondition = "(" + choiceCondition + ") || (" + condition + ")";
        }
    }
    
    /**
     * Creates and returns the additional depends condition that is needed to ensure that the model is not broken
     * when all choice items can't fulfill their conditions.
     * @return A {@link RSFDependsCondition} or <code>null</code> if no conditions have been added.
     */
    RSFDependsCondition getAdditionalDependsCondition() {
        RSFDependsCondition result = null;
        
        List<String> choiceItemConditions = new ArrayList<String>();
        for (List<String> conditions : choiceItems.values()) {
            choiceItemConditions.addAll(conditions);
        }
        
        if (choiceItemConditions.size() > 0) {
            StringBuffer condition = new StringBuffer();
            if (choiceCondition != null) {
                condition.append("\"(");
                condition.append(choiceCondition);
                condition.append(") && (");
            } else {
                condition.append("\"(");
            }
            
            for (int i = 0; i < choiceItemConditions.size(); i++) {
                String choiceItemCondition = choiceItemConditions.get(i);
                boolean addBrackets = choiceItemCondition.contains("||") || choiceItemCondition.contains("&&");
                if (addBrackets) {
                    condition.append("(");
                }
                condition.append(choiceItemCondition);
                if (addBrackets) {
                    condition.append(")");
                }
                if (i != choiceItemConditions.size() - 1) {
                    condition.append(" || ");
                }
            }
            condition.append(")\"");
            result = new RSFDependsCondition(choiceItem, condition.toString());
        }
        
        return result;
    }
    
    @Override
    List<ConstraintSyntaxTree> toPureBooleanConstraintSyntaxTree(RSFReader reader) throws ParserException {
        List<ConstraintSyntaxTree> trees = new ArrayList<ConstraintSyntaxTree>();
        
        ConstraintSyntaxTree choiceVar = getSelectedVariable(choiceItem, reader);
        ConstraintSyntaxTree notChoiceVar = getUnselectedVariable(choiceItem, reader);
        
        // Create one tree with all variables OR'd together (to make sure that at least one is selected if choiceVar
        //   is true)
        ConstraintSyntaxTree allItemsTree = notChoiceVar;
        
        RSFItem[] choiceItemsArray = choiceItems.keySet().toArray(new RSFItem[] {});
        
        // And for each unique combination of choiceItems:
        for (int i = 0; i < choiceItemsArray.length; i++) {
            ConstraintSyntaxTree var1 = getSelectedVariable(choiceItemsArray[i], reader);
            ConstraintSyntaxTree notVar1 = getUnselectedVariable(choiceItemsArray[i], reader);
            
            // Only include choice items in the XOR, which do not depend on other choice items
            if (!dependsOnOtherChoiceItems(choiceItemsArray[i])) {
            
                for (int j = i + 1; j < choiceItemsArray.length; j++) {
                    // Only include choice items in the XOR, which do not depend on other choice items
                    if (!dependsOnOtherChoiceItems(choiceItemsArray[j])) {
                        
                        // Add the 2 variables negated and OR'd together:
                        //  (to make sure that only one or less vars are selected)
                        ConstraintSyntaxTree notVar2 = getUnselectedVariable(
                                choiceItemsArray[j], reader);
                        
                        trees.add(new OCLFeatureCall(notVar1, OclKeyWords.OR, notVar2));
                        
                    }
                }
                
                allItemsTree = new OCLFeatureCall(allItemsTree, OclKeyWords.OR, var1);
            
            }
            
            // Add choiceVar or not var1 (to make sure that no variables are selected if choiceVar is false)
            trees.add(new OCLFeatureCall(choiceVar, OclKeyWords.OR, notVar1));
        }
        
        trees.add(allItemsTree);
        
        if (choiceItem.getDatatype().equals(Datatype.TRISTATE)) {
            trees.addAll(getModulePart(reader));
        }
        
        return trees;
    }
    
    /**
     * Checks whether the given choice item depends on any other choice item.
     * 
     * @param item The choice item that may depend on other choice items.
     * @return <code>true</code> if the given choice item depends on other choice items.
     */
    private boolean dependsOnOtherChoiceItems(RSFItem item) {
        List<String> constraints = choiceItems.get(item);
        
        boolean depends = false;
        
        if (constraints == null) {
            LOGGER.error("dependsOnOtherChoiceItems() called with non-choice item");
        } else {
            for (String constraint : constraints) {
                if (containsChoiceItems(constraint, item.getName())) {
                    depends = true;
                    break;
                }
            }
        }
        
        return depends;
    }
    
    /**
     * Checks whether the given constraint contains any choice item.
     * 
     * @param constraint The constraint to be checked.
     * @param exclude A choice item name to exclude from the list of choice items.
     * @return <code>true</code> if the constraint contains any choice item.
     */
    private boolean containsChoiceItems(String constraint, String exclude) {
        boolean contains = false;
        
        for (RSFItem item : choiceItems.keySet()) {
            String name = item.getName();
            if (name.equals(exclude)) {
                continue;
            }
            if (constraint.contains(name)) { // TODO
                contains = true;
                break;
            }
        }
        
        return contains;
    }
    
    /**
     * Creates the additional constraints needed when the choice is a tristate.
     * @param reader The reader to get variables from.
     * @return A list with all conditions.
     */
    private List<ConstraintSyntaxTree> getModulePart(RSFReader reader) {
        List<ConstraintSyntaxTree> trees = new ArrayList<ConstraintSyntaxTree>();
        
        ConstraintSyntaxTree notChoiceVar = getUnselectedVariable(choiceItem, reader);
        
        ConstraintSyntaxTree choiceModuleVar = varPool.obtainVariable(
                reader.getVariable(choiceItem.getName() + "_MODULE"));
        ConstraintSyntaxTree notChoiceModuleVar = new OCLFeatureCall(choiceModuleVar, OclKeyWords.NOT);
        
        // for each choiceItem
        for (RSFItem currentItem : choiceItems.keySet()) {
            if (!currentItem.getDatatype().equals(Datatype.TRISTATE)) {
                LOGGER.error("tristate choice (" + choiceItem.getName() + ") has non-tristate ChoiceItems");
            }
            ConstraintSyntaxTree notItem = getUnselectedVariable(currentItem, reader);
            ConstraintSyntaxTree itemModule = varPool.obtainVariable(
                    reader.getVariable(currentItem.getName() + "_MODULE"));
            ConstraintSyntaxTree notItemModule = new OCLFeatureCall(itemModule, OclKeyWords.NOT);
            
            trees.add(new OCLFeatureCall(choiceModuleVar, OclKeyWords.OR, notItemModule));
            trees.add(new OCLFeatureCall(notChoiceVar, OclKeyWords.OR, notItemModule));
            trees.add(new OCLFeatureCall(notChoiceModuleVar, OclKeyWords.OR, notItem));
            
        }
        
        return trees;
    }

    @Override
    List<ConstraintSyntaxTree> toNotBooleanConstraintSyntaxTree(RSFReader reader) {
        // TODO implement
        return null;
    }
    
}
