package de.uni_hildesheim.sse.trans.scenario;

import java.io.File;

import org.junit.Assert;
import org.junit.Test;

import de.uni_hildesheim.sse.trans.AllTests;
import de.uni_hildesheim.sse.trans.DimacsTestUtils;
import de.uni_hildesheim.sse.trans.convert.OptimizationParameter;

/**
 * Tests the translation of a RSF file into DIMACS format.
 * @author El-Sharkawy
 *
 */
public class RsfToDimacsTranslationTest {
    
    /**
     * Tests whether tristate variables are translated correctly.
     * @see <a href="https://projects.sse.uni-hildesheim.de/agilo/ModelTranslator/ticket/8">Requirement 8</a>
     */
    @Test
    public void testTristateVariables() {
        File input = new File(AllTests.INPUT_FOLDER, "tristateVariableTest.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        
        // Variables
        int var = DimacsTestUtils.getNumberOfVariable(result, "VARIABLE");
        int moduleVar = DimacsTestUtils.getNumberOfVariable(result, "VARIABLE_MODULE");
        
        // Conditions
        Assert.assertTrue("Tristate conditition not included",
                DimacsTestUtils.containsConstraint(result, -var, -moduleVar));
    }
    
    /**
     * Tests whether string and integer variables are translated correctly.
     */
    @Test
    public void testStringAndIntegerVariables() {
        File input = new File(AllTests.INPUT_FOLDER, "stringAndIntegerVariablesTest.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        
        // Variables
        int int5 = DimacsTestUtils.getNumberOfVariable(result, "INT_VAR=5");
        int intEmpty = DimacsTestUtils.getNumberOfVariable(result, "INT_VAR=n");
        int strHallo = DimacsTestUtils.getNumberOfVariable(result, "STR_VAR=Hallo");
        int strEmpty = DimacsTestUtils.getNumberOfVariable(result, "STR_VAR=n");
        
        int strVar = DimacsTestUtils.getNumberOfVariableNoAssert(result, "STR_VAR");
        Assert.assertTrue("String variable without comparision included", strVar == -1);
        
        int intVar = DimacsTestUtils.getNumberOfVariableNoAssert(result, "INT_VAR");
        Assert.assertTrue("Integer variable without comparision included", intVar == -1);
        
        /*
         * Test whether following conditions are included: (maximal 1 possible value is true)
         * - Not(INT_VAR=5) or Not(INT_VAR=n)
         * - Not(STR_VAR=Hallo) or Not(STR_VAR=n)
         */
        Assert.assertTrue("Error, missing: Not(INT_VAR=5) or Not(INT_VAR=n)",
                DimacsTestUtils.containsConstraint(result, -int5, -intEmpty));
        Assert.assertTrue("Error, missing: Not(STR_VAR=Hallo) or Not(STR_VAR=n)",
                DimacsTestUtils.containsConstraint(result, -strEmpty, -strHallo));
        
        /*
         * Test whether all comparison variables are found
         */
        int[] manyComp = new int[10];
        for (int i = 1; i <= 9; i++) {
            manyComp[i] = DimacsTestUtils.getNumberOfVariable(result, "STR_WITH_MANY_COMPARISONS=" + i);
        }
        
        for (int i = 1; i <= 9; i++) {
            for (int j = i + 1; j <= 9; j++) {
                Assert.assertTrue("Error, missing constraint",
                        DimacsTestUtils.containsConstraint(result, -manyComp[i], -manyComp[j]));
            }
        }
        
    }
    
    /**
     * Tests whether Strings are translated correctly.
     * This tests also includes the validation of the correct
     * translation of depends constrains.
     * A depends relation "A depends B" will be translated to "Not(B) implies Not(A)". 
     */
    @Test
    public void testEmptyStrings() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_emptyString.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();

        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int fwLoader = DimacsTestUtils.getNumberOfVariable(result, "FW_LOADER");
        int fwLoaderModule = DimacsTestUtils.getNumberOfVariable(result, "FW_LOADER_MODULE");
        int extraFirmwareEmpty = DimacsTestUtils.getNumberOfVariable(result, "EXTRA_FIRMWARE=n");
        int extraFirmwareDirEmpty = DimacsTestUtils.getNumberOfVariable(result, "EXTRA_FIRMWARE_DIR=n");
        
        /*
         * Test whether following constraints are included:
         * - Depends constraints:
         *  � EXTRA_FIRMWARE depends FW_LOADER
         *  � EXTRA_FIRMWARE_DIR depends EXTRA_FIRMWARE!=''
         * - Translated DIMACS constraints
         *  � FW_LOADER OR NOT(EXTRA_FIRMWARE)
         *  � EXTRA_FIRMWARE OR NOT(EXTRA_FIRMWARE_DIRs)
         */
        Assert.assertTrue("Error: Expected Constraint not included.",
            DimacsTestUtils.containsConstraint(result, fwLoader, fwLoaderModule, extraFirmwareEmpty));
        Assert.assertTrue("Error: Expected Constraint not included.",
            DimacsTestUtils.containsConstraint(result, -1 * extraFirmwareEmpty, extraFirmwareDirEmpty));
    }
    
    /**
     * Tests whether ItemSelect statements are translated correctly into CNF constraints.
     * Tests Boolean select Statements.
     * @see <a href="https://projects.sse.uni-hildesheim.de/agilo/ModelTranslator/ticket/9">Requirement 9</a>
     */
    @Test
    public void testItemSelectStatementsBoolean() {
        File input = new File(AllTests.INPUT_FOLDER, "ItemSelectsBoolean.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int var1 = DimacsTestUtils.getNumberOfVariable(result, "VAR1");
        int var2 = DimacsTestUtils.getNumberOfVariable(result, "VAR2");
        int var3 = DimacsTestUtils.getNumberOfVariable(result, "VAR3");
        int var4 = DimacsTestUtils.getNumberOfVariable(result, "VAR4");
        
        /* 
         * Test whether following constraint is included:
         * Not(VAR1) or VAR2
         * Not(VAR3) or Not(VAR2) or VAR4
         */
        Assert.assertTrue("Error, missing: Not(VAR1) or VAR2",
            DimacsTestUtils.containsConstraint(result, -var1, var2));
        Assert.assertTrue("Error, missing: Not(VAR3) or Not(VAR2) or VAR4",
            DimacsTestUtils.containsConstraint(result, -var3, -var2, var4));
    }
    
    /**
     * Tests whether ItemSelect statements are translated correctly into CNF constraints.
     */
    @Test
    public void testItemSelectStatements() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_ItemSelects.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int iwc3200top = DimacsTestUtils.getNumberOfVariable(result, "IWMC3200TOP");
        int iwc3200topModule = DimacsTestUtils.getNumberOfVariable(result, "IWMC3200TOP_MODULE");
        int fwLoader = DimacsTestUtils.getNumberOfVariable(result, "FW_LOADER");
        int fwLoaderModule = DimacsTestUtils.getNumberOfVariable(result, "FW_LOADER_MODULE");
        int blubb = DimacsTestUtils.getNumberOfVariable(result, "BLUBB");
        
        /* 
         * Test whether following constraint is included:
         * Not(IWMC3200TOP AND BLUBB) or FW_LOADER
         * -->
         * Not(IWMC3200TOP) OR NOT(BLUBB) or FW_LOADER
         */
        Assert.assertTrue("Error: Expected Constraint not included.",
            DimacsTestUtils.containsConstraint(result, -iwc3200top, -blubb, fwLoader));
        Assert.assertTrue("Error: Expected Constraint not included.",
                DimacsTestUtils.containsConstraint(result, -iwc3200topModule, -blubb, fwLoaderModule));
    }
    
    /**
     * Tests whether ItemSelect statements are translated correctly into CNF constraints.
     */
    @Test
    public void testBooleaItemSelects() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_BooleanItemSelects.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int a = DimacsTestUtils.getNumberOfVariable(result, "A");
        int b = DimacsTestUtils.getNumberOfVariable(result, "B");
        int c = DimacsTestUtils.getNumberOfVariable(result, "C");
        
        Assert.assertTrue("Error: Expected Constraint not included.",
            DimacsTestUtils.containsConstraint(result, -a, -b, c));
    }
    
    /**
     * Tests whether the combination of prompts and default values are considered. Tests:
     * <ul>
     * <li>Variables with default and no prompt will create a constraint for the default</li>
     * <li>Testing of defaults with conditions</li>
     * <li>Testing of prompts with conditions</li>
     * <li>Testing negation of string variables</li>
     * </ul>
     */
    @Test
    public void testDefaultsAndPrompts() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_DefaultAndPrompts.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        
        int arch32 = DimacsTestUtils.getNumberOfVariable(result, "ARCH=x86");
        int arch64 = DimacsTestUtils.getNumberOfVariable(result, "ARCH=x86_64");
        int bit = DimacsTestUtils.getNumberOfVariable(result, "64BIT");
        
        /*
         * Test whether following 3 constraints are included:
         * - ARCH='x86_64'
         * - ARCH='x86_64' implies 64BIT
         *   -> Not(ARCH='x86_64') or 64BIT
         * - ARCH='x86' implies (64BIT or !64BIT)
         *   -> tautology -> skip this constraint
         * - (!ARCH='x86_64' and !ARCH='x86') implies !64Bit (combination of prompt and default)
         *   -> ARCH='x86_64' or ARCH='x86' or !64BIT
         */
        Assert.assertTrue("Error: ARCH='x86_64' constant not included",
            DimacsTestUtils.containsConstraint(result, arch64));
        Assert.assertTrue("Error: 64Bit default not included",
            DimacsTestUtils.containsConstraint(result, -1 * arch64, bit));
        Assert.assertTrue("Error: 64Bit default and prompt not included",
            DimacsTestUtils.containsConstraint(result, arch32, arch64, -1 * bit));
    }
    
    /**
     * Tests whether choices are translated correctly into CNF constraints.
     */
    @Test
    public void testChoices() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_Choices.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        
        int choice12 = DimacsTestUtils.getNumberOfVariable(result, "CHOICE_12");
        int hz100 = DimacsTestUtils.getNumberOfVariable(result, "HZ_100");
        int hz250 = DimacsTestUtils.getNumberOfVariable(result, "HZ_250");
        int hz300 = DimacsTestUtils.getNumberOfVariable(result, "HZ_300");
        int hz1000 = DimacsTestUtils.getNumberOfVariable(result, "HZ_1000");
        int a = DimacsTestUtils.getNumberOfVariable(result, "A");
        int b = DimacsTestUtils.getNumberOfVariable(result, "B");
        int c = DimacsTestUtils.getNumberOfVariable(result, "C");
        int d = DimacsTestUtils.getNumberOfVariable(result, "D");
        
        /*
         * Test whether the following constraints are included:
         * NOT(CHOICE_12) OR NOT(HZ_100) OR NOT(HZ_250)
         * NOT(CHOICE_12) OR NOT(HZ_100) OR NOT(HZ_300)
         * NOT(CHOICE_12) OR NOT(HZ_100) OR NOT(HZ_1000)
         * (skip other combinations)
         */
        Assert.assertTrue("Error: Missing choice constraint",
            DimacsTestUtils.containsConstraint(result, -hz100, -hz250));
        Assert.assertTrue("Error: Missing choice constraint",
            DimacsTestUtils.containsConstraint(result, -hz100, -hz300));
        Assert.assertTrue("Error: Missing choice constraint",
            DimacsTestUtils.containsConstraint(result, -hz100, -hz1000));
        
        /*
         * NOT(CHOICE_12) OR HZ_100 OR HZ_250 OR HZ_300 OR HZ_1000
         */
        Assert.assertTrue("Error: Missing choice constraint",
            DimacsTestUtils.containsConstraint(result, -choice12, hz100, hz250, hz300, hz1000));
        
        /*
         * CHOICE_12 OR NOT(HZ_100)
         * CHOICE_12 OR NOT(HZ_250)
         * (skip other variables)
         */
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, choice12, -hz100));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, choice12, -hz250));
        
        /*
         * Check if the following additional constraint is included:
         * - depends  CHOICE  (dependsOf(CHOICE) &&
         *      (dependsOf(hz100) || dependsOf(hz250) || ependsOf(hz300) || ependsOf(hz1000))
         * = depends  CHOICE  (A || B || C || D)
         * -> Not(CHOICE) or A or B or C or D
         */
        Assert.assertTrue("Error: Missing constraint", 
                DimacsTestUtils.containsConstraint(result, -choice12, a, b, c, d));
    }
    
    /**
     * Tests whether choices that contain choice items that depend on each other
     * are translated correctly into CNF constraints.
     */
    @Test
    public void testChoicesDependOnChoices() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_ChoicesDependOnChoices.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        
        int choice = DimacsTestUtils.getNumberOfVariable(result, "CHOICE");
        int a = DimacsTestUtils.getNumberOfVariable(result, "A");
        int b = DimacsTestUtils.getNumberOfVariable(result, "B");
        int c = DimacsTestUtils.getNumberOfVariable(result, "C");
        int d = DimacsTestUtils.getNumberOfVariable(result, "D");
        
        /*
         * Test whether the following constraints are included:
         * NOT(A) OR NOT(B)
         * NOT(A) OR NOT(C)
         * NOT(B) OR NOT(C)
         * 
         * NOT(CHOICE) or A OR B or C
         * 
         * CHOICE or NOT(A)
         * CHOICE or NOT(B)
         * CHOICE or NOT(C)
         * CHOICE or NOT(D)
         * 
         * NOT(D) or A
         */
        Assert.assertTrue("Error: Missing choice constraint",
                 DimacsTestUtils.containsConstraint(result, -a, -b));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, -a, -c));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, -b, -c));
        
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, -choice, a, b, c));
        
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, choice, -a));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, choice, -b));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, choice, -c));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, choice, -d));
        
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, -d, a));
        
        /*
         * Test whether the following constraints are NOT included:
         * NOT(A) or NOT(D)
         * NOT(B) or NOT(D)
         * NOT(C) or NOT(D)
         * 
         * NOT(CHOICE) or A or B or C or D
         */
        Assert.assertFalse("Error: Unwanted choice constraint included",
                 DimacsTestUtils.containsConstraint(result, -a, -d));
        Assert.assertFalse("Error: Unwanted choice constraint included",
                DimacsTestUtils.containsConstraint(result, -b, -d));
        Assert.assertFalse("Error: Unwanted choice constraint included",
                DimacsTestUtils.containsConstraint(result, -c, -d));
        
        Assert.assertFalse("Error: Unwanted choice constraint included",
                DimacsTestUtils.containsConstraint(result, -choice, a, b, c, d));
    }
    
    /**
     * Tests whether tristate choices are translated correctly into CNF.
     */
    @Test
    public void testTristateChoices() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_tristateChoices.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        
        int choice = DimacsTestUtils.getNumberOfVariable(result, "CHOICE");
        int item1 = DimacsTestUtils.getNumberOfVariable(result, "ITEM1");
        int item2 = DimacsTestUtils.getNumberOfVariable(result, "ITEM2");
        int choiceModule = DimacsTestUtils.getNumberOfVariable(result, "CHOICE_MODULE");
        int item1Module = DimacsTestUtils.getNumberOfVariable(result, "ITEM1_MODULE");
        int item2Module = DimacsTestUtils.getNumberOfVariable(result, "ITEM2_MODULE");
        
        Assert.assertTrue("Error: Missing constraint",
                DimacsTestUtils.containsConstraint(result, choiceModule, -item1Module));
        Assert.assertTrue("Error: Missing constraint",
                DimacsTestUtils.containsConstraint(result, choiceModule, -item2Module));
        
        Assert.assertTrue("Error: Missing constraint",
                DimacsTestUtils.containsConstraint(result, -choice, -item1Module));
        Assert.assertTrue("Error: Missing constraint",
                DimacsTestUtils.containsConstraint(result, -choice, -item2Module));
        
        Assert.assertTrue("Error: Missing constraint",
                DimacsTestUtils.containsConstraint(result, -choiceModule, -item1));
        Assert.assertTrue("Error: Missing constraint",
                DimacsTestUtils.containsConstraint(result, -choiceModule, -item2));
    }
    
    /**
     * Tests single depends statements. Tests
     * <ul>
     * <li>Boolean variable depends Boolean variable</li>
     * <li>Tristate variable depends Tristate variable</li>
     * </ul>
     * @see <a href="https://projects.sse.uni-hildesheim.de/agilo/ModelTranslator/ticket/3">Requirement 3</a>
     */
    @Test
    public void testSingleDepends() {
        File input = new File(AllTests.INPUT_FOLDER, "SingleDependsTest.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int bool1 = DimacsTestUtils.getNumberOfVariable(result, "BOOL1");
        int bool2 = DimacsTestUtils.getNumberOfVariable(result, "BOOL2");
        int bool3 = DimacsTestUtils.getNumberOfVariable(result, "BOOL3");
        int bool4 = DimacsTestUtils.getNumberOfVariable(result, "BOOL4");
        int tri1 = DimacsTestUtils.getNumberOfVariable(result, "TRI1");
        int tri1Module = DimacsTestUtils.getNumberOfVariable(result, "TRI1_MODULE");
        int tri2 = DimacsTestUtils.getNumberOfVariable(result, "TRI2");
        int tri2Module = DimacsTestUtils.getNumberOfVariable(result, "TRI2_MODULE");
        int tri3 = DimacsTestUtils.getNumberOfVariable(result, "TRI3");
        int tri3Module = DimacsTestUtils.getNumberOfVariable(result, "TRI3_MODULE");
        int tri4 = DimacsTestUtils.getNumberOfVariable(result, "TRI4");
        int tri4Module = DimacsTestUtils.getNumberOfVariable(result, "TRI4_MODULE");
        
        /*
         * Test boolean depends on boolean:
         * - BOOL1 depends from BOOL2
         *  -> BOOL2 or Not(BOOL1)
         */
        Assert.assertTrue("Error, missing: BOOL2 or Not(BOOL1)",
            DimacsTestUtils.containsConstraint(result, -bool1, bool2));
        
        /*
         * Test boolean depends on tristate:
         * - BOOL3 depends from TRI3
         *  -> TRI3 or TRI3_MODULe or Not(BOOL3)
         */
        Assert.assertTrue("Error, missing: TRI3 or TRI3_MODULE or Not(BOOL3)", 
                DimacsTestUtils.containsConstraint(result, tri3, tri3Module, -bool3));
        
        /*
         * Test tristate depends on boolean:
         * - TRI4 depends from BOOL4
         *  -> BOOL4 or Not(TRI4)
         *  -> BOOL4 or Not(TRI4_MODULE)
         */
        Assert.assertTrue("Error, missing: BOOL4 or Not(TRI4)", 
                DimacsTestUtils.containsConstraint(result, bool4, -tri4));
        Assert.assertTrue("Error, missing: BOOL4 or Not(TRI4_MODULE)", 
                DimacsTestUtils.containsConstraint(result, bool4, -tri4Module));
        
        /*
         * Test tristate depends on tristate:
         * - TRI1 depends from TRI2
         *  -> Not(TRI1) or TRI2 or TRI2_MODULE
         *  -> Not(TRI1_MODULE) or TRI2 or TRI2_MODULE
         */
        Assert.assertTrue("Error, missing: Not(TRI1) or TRI2 or TRI2_MODULE",
                DimacsTestUtils.containsConstraint(result, -tri1, tri2, tri2Module));
        Assert.assertTrue("Error, missing: Not(VAR3_MODULE) or VAR4 or VAR4_MODULE",
                DimacsTestUtils.containsConstraint(result, -tri1Module, tri2, tri2Module));
    }
    
    /**
     * Tests whether a variable can have more than one depends statement.
     * All depends statements must be combined using an OR expression, e.g.:
     * <pre><code>
     * A depends B
     * A depends C
     * </code></pre>
     * must be translated to
     * <pre><code>
     * A depends (B OR C)
     * </code></pre>
     * @see <a href="https://projects.sse.uni-hildesheim.de/agilo/ModelTranslator/ticket/4">Requirement 4</a>
     */
    @Test
    public void testMultipleDepends() {
        File input = new File(AllTests.INPUT_FOLDER, "MultipleDependsTest.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int x32 = DimacsTestUtils.getNumberOfVariable(result, "X86_32");
        int x64 = DimacsTestUtils.getNumberOfVariable(result, "X86_64");
        int extendend = DimacsTestUtils.getNumberOfVariable(result, "X86_EXTENDED_PLATFORM");
        
        /*
         * Test whether the following constraints are included:
         * - NOT(X86_32) OR NOT(X86_64)
         * - NOT(X86_32 OR X86_64) implies NOT(X86_EXTENDED_PLATFORM)
         *  -> X86_32 OR X86_64 OR NOT(X86_EXTENDED_PLATFORM)
         */
        Assert.assertTrue("Error, missing: NOT(X86_32) OR NOT(X86_64)",
            DimacsTestUtils.containsConstraint(result, -1 * x32, -1 * x64));
        Assert.assertFalse("Error, disallowed constraitn found: NOT(X86_EXTENDED_PLATFORM) OR X86_32",
            DimacsTestUtils.containsConstraint(result, -1 * extendend, x32));
        Assert.assertFalse("Error, disallowed constraitn found: NOT(X86_EXTENDED_PLATFORM) OR X86_64",
            DimacsTestUtils.containsConstraint(result, -1 * extendend, x64));
        Assert.assertTrue("Error, missing: NOT(X86_EXTENDED_PLATFORM) OR X86_32 OR X86_64",
            DimacsTestUtils.containsConstraint(result, -1 * extendend, x32, x64));
    }

    /**
     * Tests whether prompt and default will not lead to a constraint.
     */
    @Test
    public void testPromptAndDefault() {
        File input = new File(AllTests.INPUT_FOLDER, "PromptAndDefaultTest.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        
        // Model has one variable with prompt and default -> Thus, model must not contain a constraint
        Assert.assertTrue("Error; Model must not contain any constraints!", result.contains("p CNF 1 0"));
    }
    
    /**
     * Tests whether a default value becomes true when a prompt disappears.
     */
    @Test
    public void testConditionalPromptAndDefault() {
        File input = new File(AllTests.INPUT_FOLDER, "ConditionalPromptAndDefaultTest.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        
        int var1 = DimacsTestUtils.getNumberOfVariable(result, "VAR1");
        int var2 = DimacsTestUtils.getNumberOfVariable(result, "VAR2");
        int var3 = DimacsTestUtils.getNumberOfVariable(result, "VAR3");
        int var4 = DimacsTestUtils.getNumberOfVariable(result, "VAR4");
        
        Assert.assertTrue("Error: Expected Constraint not included.",
            DimacsTestUtils.containsConstraint(result, var2, -var4, -var1, var3));
        Assert.assertTrue("Error: Expected Constraint not included.",
            DimacsTestUtils.containsConstraint(result, var2, -var4, var1, -var3));
    }
    
    /**
     * Tests whether the variable 4KSTACKS is translated correctly.
     */
    @Test
    public void test4KStack() {
        File input = new File(AllTests.INPUT_FOLDER, "test4KStack.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int var4K = DimacsTestUtils.getNumberOfVariable(result, "4KSTACKS");
        int varX86 = DimacsTestUtils.getNumberOfVariable(result, "X86_32");
        
        Assert.assertTrue("Error: Expected Constraint not included.",
            DimacsTestUtils.containsConstraint(result, varX86, -var4K));
    }
}
