package de.uni_hildesheim.sse.model_extender.convert;

import java.util.ArrayList;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;

/**
 * Helper class for parsing constraints from strings.
 * 
 * @author Adam Krafczyk
 */
public class ConstraintParser {

    /**
     * Parses the given string into a CST.
     * 
     * @param constraint The constraint string to parse.
     * @param varSource The source where to get variables from.
     * 
     * @return A CST representing the constraint in the string; not <code>null</code>.
     * 
     * @throws ConstraintParserException If parsing the constraint fails.
     */
    public static ConstraintSyntaxTree parse(String constraint, VariableSource varSource)
        throws ConstraintParserException {
        
        constraint = constraint.replace(" ", "");
        
        String[] parts = split(constraint);
        if (parts.length % 2 != 1) {
            throw new ConstraintParserException("Unbalanced constraint: \"" + constraint + "\"");
        }
        
        ConstraintSyntaxTree result = null;
        
        if (parts.length == 1) {
            String part = parts[0];
            if (part.startsWith("!")) {
                // Handle negations
                result = new OCLFeatureCall(parse(part.substring(1), varSource), OclKeyWords.NOT);
            } else if (part.startsWith("(")) {
                // Unpack brackets
                if (!part.endsWith(")")) {
                    throw new ConstraintParserException("Invalid constraint part: \"" + part + "\"");
                }
                String newPart = part.substring(1, part.length() - 1);
                result = parse(newPart, varSource);
            } else {
                // We have a variable
                result = varSource.getVariable(part);
            }
        } else {
            // Construct the constraint by combining the parts
            result = parse(parts[0], varSource);
            for (int i = 1; i < parts.length; i += 2) {
                // Recursively parse sub-parts
                ConstraintSyntaxTree newPart = parse(parts[i + 1], varSource);
                String operator = null;
                if (parts[i].equals("&&")) {
                    operator = OclKeyWords.AND;
                } else if (parts[i].equals("||")) {
                    operator = OclKeyWords.OR;
                } else {
                    throw new ConstraintParserException("Invalid operator \""
                            + parts[i] + "\" in constraint \"" + constraint + "\"");
                }
                
                result = new OCLFeatureCall(result, operator, newPart);
            }
            
        }
        
        return result;
    }
    
    
    /**
     * Splits the given constraint into parts. The even elements in the resulting array
     * contain the parts, the odd elements the connection between the parts.
     * 
     * @param constraint The constraint string to parse.
     * 
     * @return The parts of the constraint.
     * 
     * @throws ConstraintParserException If parsing the constraint fails.
     */
    private static String[] split(String constraint) throws ConstraintParserException {
        ArrayList<String> result = new ArrayList<String>();
        
        int begin = 0;
        int i = 0;
        int bracketDepth = 0;
        
        for (; i < constraint.length(); i++) {
            char currentChar = constraint.charAt(i);
            
            // Update bracketDepth if we found a bracket
            if (currentChar == '(') {
                bracketDepth++;
            } else if (currentChar == ')') {
                bracketDepth--;
            }
            if (bracketDepth < 0) {
                throw new ConstraintParserException("Unbalanced brackets in constraint \"" + constraint + "\"");
            }
            
            // If we are not in a bracket
            if (bracketDepth == 0) {
                // Possibly split at the operator
                if (currentChar == '|') {
                    char nextChar = constraint.charAt(i + 1);
                    if (nextChar != '|') {
                        throw new ConstraintParserException("Invalid operation \"|"
                                + nextChar + "\" in constraint \"" + constraint + "\"");
                    }
                    
                    result.add(constraint.substring(begin, i));
                    result.add("||");
                    begin = i + 2;
                    i++;
                } else if (currentChar == '&') {
                    char nextChar = constraint.charAt(i + 1);
                    if (nextChar != '&') {
                        throw new ConstraintParserException("Invalid operation \"&"
                                + nextChar + "\" in constraint \"" + constraint + "\"");
                    }
                    
                    result.add(constraint.substring(begin, i));
                    result.add("&&");
                    begin = i + 2;
                    i++;
                }
            }
        }
        
        if (bracketDepth != 0) {
            throw new ConstraintParserException("Unbalanced brackets in constraint \"" + constraint + "\"");
        }
        
        if (begin < i) {
            result.add(constraint.substring(begin, i));
        }
        
        for (String s : result) {
            if (s.length() < 1) {
                throw new ConstraintParserException("Missing variable name in constraint \"" + constraint + "\"");
            }
        }
        
        return result.toArray(new String[] {});
    }
    
}
