package de.uni_hildesheim.see.model_extender;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.HashMap;

import org.junit.Test;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.cst.VariablePool;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.datatypes.BooleanType;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.model_extender.convert.ConstraintParser;
import de.uni_hildesheim.sse.model_extender.convert.ConstraintParserException;
import de.uni_hildesheim.sse.model_extender.convert.VariableSource;

/**
 * Contains tests for the {@link ConstraintParser}.
 * @author Adam Krafczyk
 */
public class ConstraintParserTest {

    /**
     * A small dummy variable source.
     * @author Adam Krafczyk
     */
    private static class DummyVariableSource implements VariableSource {

        private HashMap<String, AbstractVariable> varDecls = new HashMap<String, AbstractVariable>();
        private VariablePool varPool = new VariablePool();
        
        @Override
        public Variable getVariable(String name) {
            AbstractVariable decl = varDecls.get(name);
            if (decl == null) {
                decl = new DecisionVariableDeclaration(name, BooleanType.TYPE, null);
                varDecls.put(name, decl);
            }
            return varPool.obtainVariable(decl);
        }
        
    }
    
    /**
     * Test whether unbalanced constraint are found correctly.
     */
    @Test
    public void testOperatorBalancing() {
        try {
            ConstraintParser.parse("A && B", new DummyVariableSource());
            ConstraintParser.parse("A", new DummyVariableSource());
        } catch (ConstraintParserException e) {
            fail("This constraint should be parseable");
        }
        
        try {
            ConstraintParser.parse("A &&", new DummyVariableSource());
            fail("This constraint should not be parseable");
        } catch (ConstraintParserException e) {
        }
    }
    
    /**
     * Tests whether invalid operations are found correctly.
     */
    @Test
    public void testInvalidOperations() {
        try {
            ConstraintParser.parse("A && B", new DummyVariableSource());
            ConstraintParser.parse("A || B", new DummyVariableSource());
            ConstraintParser.parse("A || B && C || D && E", new DummyVariableSource());
        } catch (ConstraintParserException e) {
            fail("This constraint should be parseable");
        }
        
        try {
            ConstraintParser.parse("A & B", new DummyVariableSource());
            fail("This constraint should not be parseable");
        } catch (ConstraintParserException e) {
        }
        
        try {
            ConstraintParser.parse("A | B", new DummyVariableSource());
            fail("This constraint should not be parseable");
        } catch (ConstraintParserException e) {
        }
    }
    
    /**
     * Tests whether unbalanced brackets are found correctly.
     */
    @Test
    public void testBracketBalancing() {
        try {
            ConstraintParser.parse("(A && B)", new DummyVariableSource());
            ConstraintParser.parse("(A) || (B && (C || D))", new DummyVariableSource());
        } catch (ConstraintParserException e) {
            fail("This constraint should be parseable");
        }
        
        try {
            ConstraintParser.parse("(A || B", new DummyVariableSource());
            fail("This constraint should not be parseable");
        } catch (ConstraintParserException e) {
        }
        
        try {
            ConstraintParser.parse("A || B)", new DummyVariableSource());
            fail("This constraint should not be parseable");
        } catch (ConstraintParserException e) {
        }
        
        try {
            ConstraintParser.parse("(A || (B && C)", new DummyVariableSource());
            fail("This constraint should not be parseable");
        } catch (ConstraintParserException e) {
        }
    }
    
    /**
     * Tests whether missing variable names are found correctly.
     */
    @Test
    public void testEmptyVariables() {
        try {
            ConstraintParser.parse("(A || (&& C))", new DummyVariableSource());
            fail("This constraint should not be parseable");
        } catch (ConstraintParserException e) {
        }
    }
    
    /**
     * Tests whether constraints are constrcuted correctly.
     * 
     * @throws ConstraintParserException Should not occur.
     */
    @Test
    public void testConstraints() throws ConstraintParserException {
        DummyVariableSource varSource = new DummyVariableSource();
        
        ConstraintSyntaxTree tree = ConstraintParser.parse("A && (!(B || !(C && !D)) || !(E>7))", varSource);
        
        OCLFeatureCall call = (OCLFeatureCall) tree;
        assertEquals(OclKeyWords.AND, call.getOperation());
        
        assertTrue(call.getOperand().equals(varSource.getVariable("A"))
                || call.getParameter(0).equals(varSource.getVariable("A")));
        
        // tree.accept(new DebugConstraintTreeVisitor());
        
        // TODO: completely check tree
    }

}
