package de.uni_hildesheim.sse.trans.in.rsf;

import java.util.ArrayList;
import java.util.List;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.cst.VariablePool;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.Main;
import de.uni_hildesheim.sse.trans.in.ParserException;
import de.uni_hildesheim.sse.trans.in.ParserException.ParserExceptionType;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Represents a condition read from a .rsf file.
 * 
 * @author Adam Krafczyk
 */
abstract class RSFCondition {
    protected static VariablePool varPool = new VariablePool();
    
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(RSFCondition.class, Main.ID);
    
    /**
     * Returns this condition as a pure boolean expression.
     * @param reader The reader to get the variables from.
     * @return A List of {@link ConstraintSyntaxTree}s representing this expression. The parts should be handled as if
     *         combined with AND. empty if expression is optimized out.
     * @throws ParserException If the conditions contains an operation, that is not supported by the parser.
     */
    abstract List<ConstraintSyntaxTree> toPureBooleanConstraintSyntaxTree(RSFReader reader) throws ParserException;
    
    /**
     * Returns this condition as a {@link ConstraintSyntaxTree}.
     * @param reader The reader to get the variables from.
     * @return A List of {@link ConstraintSyntaxTree}s representing this expression. The parts should be handled as if
     *         combined with AND. empty if expression is optimized out.
     */
    abstract List<ConstraintSyntaxTree> toNotBooleanConstraintSyntaxTree(RSFReader reader);
    
    /**
     * Converts the given variable from a string to a {@link ConstraintSyntaxTree}.
     * A ! at the beginning will result in a {@link OCLFeatureCall} with NOT.
     * @param reader The reader to read variable declarations from.
     * @param var The string representing a variable.
     * @param originalVariable The variable that depends on the condition (used to handle 'm').
     * @return The variable as {@link ConstraintSyntaxTree}.
     */
    ConstraintSyntaxTree getVariable(RSFReader reader, String var, String originalVariable) {
        ConstraintSyntaxTree result = null;
        // Remove parenthesis around variable names
        if (var.startsWith("(")) {
            var = var.substring(1);
            if (var.endsWith(")")) {
                var = var.substring(0, var.length() - 1);
            }
        }
        
        if (var.startsWith("!")) {
            var = var.substring(1);
            ConstraintSyntaxTree variable = getVariable(reader, var, originalVariable);
            result = new OCLFeatureCall(variable, OclKeyWords.NOT);
        }
        
        if (result == null) {
            result = handleConstantInVariable(reader, var, originalVariable);
        }
        
        if (result == null) {
            result = handleModuleInVariable(reader, var, originalVariable);
        }
        
        if (result == null) {
            result = handleStringInVariable(reader, var, originalVariable);
        }
        
        if (result == null) {
            if (var.contains("=")) {
                String originalVar = var.substring(0, var.indexOf('='));
                RSFItem originalItem = reader.getItemNoCreate(originalVar);
                if (originalItem.getDatatype() == Datatype.BOOLEAN
                        || originalItem.getDatatype() == Datatype.TRISTATE) {
                    LOGGER.error("Boolean or tristate variable contains '=': " + var);
                }
            }
            
            // Remove ' around variable names
            if (var.startsWith("'")) {
                var = var.substring(1);
            }
            if (var.endsWith("'")) {
                var = var.substring(0, var.length() - 1);
            }
            
            RSFItem rsfItem = reader.getItemNoCreate(var);
            
            if (rsfItem != null && rsfItem.getDatatype().equals(Datatype.TRISTATE)) {
                Variable variable = varPool.obtainVariable(reader.getVariable(var));
                Variable moduleVariable = varPool.obtainVariable(reader.getVariable(var + "_MODULE"));
                result = new OCLFeatureCall(variable, OclKeyWords.OR, moduleVariable);
            } else {
                Variable variable = varPool.obtainVariable(reader.getVariable(var));
                result = variable;
                
                int index = var.indexOf('=');
                if (index != -1) {
                    String originalVar = var.substring(0, index);
                    RSFItem originalItem = reader.getItemNoCreate(originalVar);
                    if (originalItem != null) {
                        originalItem.addComparisonVariable(variable);
                    } else {
                        // TODO
                        LOGGER.debug("RSFItem for string or int comparison missing");
                    }
                }
            }
        }
        return result;
    }
    
    /**
     * Handles constants (='y' etc.) in a variable.
     * @param reader The reader to read variable declarations from.
     * @param var The string representing a variable.
     * @param originalVariable The variable that depends on the condition.
     * @return A {@link ConstraintSyntaxTree} if the variable contained a constant expression; <code>null</code>
     *         otherwise
     */
    private ConstraintSyntaxTree handleConstantInVariable(RSFReader reader, String var, String originalVariable) {
        ConstraintSyntaxTree result = null;
        if (var.endsWith("!='y'")) {
            var = var.substring(0, var.length() - "!='y'".length());
            result = getVariable(reader, "!" + var, originalVariable);
            
        } else if (var.endsWith("='y'")) {
            var = var.substring(0, var.length() - "='y'".length());
            result = getVariable(reader, var, originalVariable);
            
        } else if (var.endsWith("!='n'")) {
            var = var.substring(0, var.length() - "!='n'".length());
            result = getVariable(reader, var, originalVariable);
            
        } else if (var.endsWith("='n'")) {
            var = var.substring(0, var.length() - "='n'".length());
            result = getVariable(reader, "!" + var, originalVariable);
            
        }
        return result;
    }
    
    /**
     * Handles modules (='m', 'm' etc.) in a variable.
     * @param reader The reader to read variable declarations from.
     * @param var The string representing a variable.
     * @param originalVariable The variable that depends on the condition.
     * @return A {@link ConstraintSyntaxTree} if the variable contained a module expression; <code>null</code>
     *         otherwise
     */
    private ConstraintSyntaxTree handleModuleInVariable(RSFReader reader, String var, String originalVariable) {
        ConstraintSyntaxTree result = null;
        if (var.endsWith("!='m'")) {
            var = var.substring(0, var.length() - "!='m'".length());
            result = getVariable(reader, "!" + var + "_MODULE", originalVariable);
            
        } else if (var.endsWith("='m'")) {
            var = var.substring(0, var.length() - "='m'".length());
            result = getVariable(reader, var + "_MODULE", originalVariable);
            
        } else if (var.equals("'m'")) {
            result = getVariable(reader, originalVariable + "_MODULE", originalVariable);
        }
        return result;
    }
    
    /**
     * Handles strings (='Foo', !='' etc.) in a variable.
     * @param reader The reader to read variable declarations from.
     * @param var The string representing a variable.
     * @param originalVariable The variable that depends on the condition.
     * @return A {@link ConstraintSyntaxTree} if the variable contained a string comparison; <code>null</code>
     *         otherwise
     */
    private ConstraintSyntaxTree handleStringInVariable(RSFReader reader, String var, String originalVariable) {
        ConstraintSyntaxTree result = null;
        if (var.contains("!=''")) {
            int index = var.indexOf("!=''");
            var = var.substring(0, index);
            
            result = getVariable(reader, "!" + var + "=n", originalVariable);
            
        } else if (var.contains("=''")) {
            int index = var.indexOf("=''");
            var = var.substring(0, index);
            
            result = getVariable(reader, var + "=n", originalVariable);
            
        } else if (var.contains("!='")) {
            StringBuffer buffer = new StringBuffer(var);
            int index;
            while ((index = buffer.indexOf("'")) != -1) {
                buffer.deleteCharAt(index);
            }
            
            // move the ! to the beginning
            index = buffer.indexOf("!=");
            buffer.deleteCharAt(index);
            
            result = getVariable(reader, "!" + buffer.toString(), originalVariable);
            
        } else if (var.contains("='")) {
            StringBuffer buffer = new StringBuffer(var);
            int index;
            while ((index = buffer.indexOf("'")) != -1) {
                buffer.deleteCharAt(index);
            }
            result = getVariable(reader, buffer.toString(), originalVariable);
        }
        return result;
    }
    
    /**
     * Converts the condition into a pure boolean {@link ConstraintSyntaxTree}.
     * @param reader The reader to get the variables from.
     * @param condition A string representing the condition as read from a .rsf file.
     * @param variable The variable that depends on the condition (used to handle 'm').
     * @return A {@link ConstraintSyntaxTree} representing the condition. <code>null</code> if always true.
     * @throws ParserException If the conditions contains an operation, that is not supported by the parser.
     */
    protected ConstraintSyntaxTree getPureBooleanConstraintSyntaxTree(RSFReader reader, String condition,
        String variable) throws ParserException {
        
        condition = condition.replace(" ", "");
        
        // We assume that ! can only be in front of variables
        
        String highestOperand = null;
        int highestOperandLevel = Integer.MAX_VALUE;
        int highestOperandPos = -1;
        int currentLevel = 0;
        
        for (int i = 0; i < condition.length(); i++) {
            char c = condition.charAt(i);
            switch (c) {
            case '(':
                currentLevel++;
                break;
            case ')':
                currentLevel--;
                break;
                
            case '&':
                if (condition.charAt(i + 1) != '&') {
                    throw new ParserException(ParserExceptionType.NOT_SUPPORTED_FIELD); // TODO
                }
                if (highestOperandLevel > currentLevel) {
                    highestOperand = OclKeyWords.AND;
                    highestOperandPos = i;
                    highestOperandLevel = currentLevel;
                }
                i++;
                break;
            
            case '|':
                if (condition.charAt(i + 1) != '|') {
                    throw new ParserException(ParserExceptionType.NOT_SUPPORTED_FIELD); // TODO
                }
                if (highestOperandLevel > currentLevel) {
                    highestOperand = OclKeyWords.OR;
                    highestOperandPos = i;
                    highestOperandLevel = currentLevel;
                }
                i++;
                break;
            
            default:
                break;
            }
        }
        
        ConstraintSyntaxTree result = null;
        
        if (highestOperandPos == -1) {
            if (!condition.equals("y")) {
                // we only have a variable
                result = getVariable(reader, condition, variable);
            }
            // result = null if condition is "y"
        } else {
            StringBuffer left = new StringBuffer();
            StringBuffer right = new StringBuffer();
            split(condition, highestOperandPos, highestOperandLevel, left, right);
            result = new OCLFeatureCall(getPureBooleanConstraintSyntaxTree(reader, left.toString(), variable),
                    highestOperand, getPureBooleanConstraintSyntaxTree(reader, right.toString(), variable));
        }
        
        return result;
    }
    
    /**
     * Converts the condition into a {@link ConstraintSyntaxTree}.
     * @param reader The reader to get the variables from.
     * @param condition A string representing the condition as read from a .rsf file.
     * @param variable The variable that depends on the condition (used to handle 'm').
     * @return A {@link ConstraintSyntaxTree} representing the condition. <code>null</code> if always true.
     */
    protected ConstraintSyntaxTree getNotBooleanConstraintSyntaxTree(RSFReader reader, String condition,
            String variable) {
        // TODO implement
        return null;
    }
    
    /**
     * Splits the given line at the given operand position into a left and right part.
     * @param line The whole line
     * @param operandPos The position of the operand in the line
     * @param operandLevel The logic level of the operand (by counting brackets)
     * @param left A {@link StringBuffer} where the left side of the operand will be written to
     * @param right A {@link StringBuffer} where the right side of the operand will be written to
     */
    private void split(String line, int operandPos, int operandLevel, StringBuffer left, StringBuffer right) {
        left.append(line.substring(0, operandPos));
        right.append(line.substring(operandPos + 2, line.length()));
        int currentLevel = 0;
        for (int i = 0; i < left.length(); i++) {
            char c = left.charAt(i);
            if ((c == '(' || c == ')') && currentLevel < operandLevel) {
                left.replace(i, i + 1, "");
            }
            if (c == '(') {
                currentLevel++;
            }
            if (c == ')') {
                currentLevel--;
            }
        }
        currentLevel = 0;
        for (int i = right.length() - 1; i >= 0; i--) {
            char c = right.charAt(i);
            if ((c == '(' || c == ')') && currentLevel < operandLevel) {
                right.replace(i, i + 1, "");
            }
            if (c == ')') {
                currentLevel++;
            }
            if (c == '(') {
                currentLevel--;
            }
        }
    }
    
    /**
     * Returns a {@link ConstraintSyntaxTree} with the expression that the variable is not set.
     * This method is meant to handle HEX, INTEGER and STRING correctly. It does not handle _MODULE variables of
     * tristates.
     * @param variable The variable that should be not set.
     * @param reader The reader to get varible declarations from.
     * @return A {@link ConstraintSyntaxTree}.
     */
    protected ConstraintSyntaxTree getUnselectedVariable(RSFItem variable, RSFReader reader) {
        ConstraintSyntaxTree notVar = null;
        
        switch (variable.getDatatype()) {
        case HEX:
        case INTEGER:
        case STRING:
            notVar = getVariable(reader, variable.getName() + "=n", null);
            break;
        default:
            DecisionVariableDeclaration varDecl = reader.getVariable(variable.getName());
            notVar = new OCLFeatureCall(varPool.obtainVariable(varDecl), OclKeyWords.NOT);
            break;
        }
        
        return notVar;
    }
    
    /**
     * Returns a {@link ConstraintSyntaxTree} with the expression that the variable is set.
     * This method is meant to handle HEX, INTEGER and STRING correctly. It does not handle _MODULE variables of
     * tristates.
     * @param variable The variable that should be set.
     * @param reader The reader to get varible declarations from.
     * @return A {@link ConstraintSyntaxTree}.
     */
    protected ConstraintSyntaxTree getSelectedVariable(RSFItem variable, RSFReader reader) {
        ConstraintSyntaxTree var = null;
        
        switch (variable.getDatatype()) {
        case HEX:
        case INTEGER:
        case STRING:
            var = new OCLFeatureCall(getUnselectedVariable(variable, reader), OclKeyWords.NOT);
            break;
        default:
            DecisionVariableDeclaration varDecl = reader.getVariable(variable.getName());
            var = varPool.obtainVariable(varDecl);
            break;
        }
        
        return var;
    }
    
    /**
     * Creates a condition to ensure that at most one of the variables is selected.
     * @param variables A not modified {@link List} of variables
     * @return A List of {@link ConstraintSyntaxTree}s representing this expression. The parts should be handled as if
     *         combined with AND. empty if expression is optimized out.
     */
    static List<ConstraintSyntaxTree> createComparisonVariableCondition(List<Variable> variables) {
        List<ConstraintSyntaxTree> result = new ArrayList<ConstraintSyntaxTree>();
        
        for (int i = 0; i < variables.size();  i++) {
            ConstraintSyntaxTree notVar1 = new OCLFeatureCall(variables.get(i), OclKeyWords.NOT);
            for (int j = i + 1; j < variables.size(); j++) {
                ConstraintSyntaxTree notVar2 = new OCLFeatureCall(variables.get(j), OclKeyWords.NOT);
                result.add(new OCLFeatureCall(notVar1, OclKeyWords.OR, notVar2));
            }
        }
        
        return result;
    }
    
}
