package de.uni_hildesheim.sse.model_extender;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.IvmlKeyWords;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.model.varModel.filter.ConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder.VisibilityType;
import de.uni_hildesheim.sse.model.varModel.filter.FilterType;
import de.uni_hildesheim.sse.model_extender.convert.ConstraintParserException;
import de.uni_hildesheim.sse.model_extender.convert.ModelExtender;
import de.uni_hildesheim.sse.model_extender.in.DimacsReader;
import de.uni_hildesheim.sse.model_extender.in.MalformedFileException;
import de.uni_hildesheim.sse.persistency.IVMLWriter;
import de.uni_hildesheim.sse.trans.out.DimacsWriter;

/**
 * Main class for running the ModelExtender.
 * 
 * @author Adam Krafczyk
 * @author El-Sharkawy
 */
public class Main {
	
	private static final boolean DEBUG = false;
		
		
    /**
     * Will extends the <tt>sourceDimacsFile</tt> with the given <tt>constraint</tt> and save the extended file to
     * <tt>destinationDimacsFile</tt>.
     * @param sourceDimacsFile The translated KConfig model in dimacs format. Must not be <tt>null</tt>.
     * @param constraint The constraint to append to the already existing model.
     * @param destinationDimacsFile The destination of the extended model. The path must exist.
     * @throws FileNotFoundException If the <tt>sourceDimacsFile</tt> was not found or could not be read.
     * @throws IOException If reading the <tt>sourceDimacsFile</tt> throws an {@link IOException}
     * or if writing the <tt>destinationDimacsFile</tt> throws an {@link IOException} 
     * @throws MalformedFileException If the <tt>sourceDimacsFile</tt> is not correctly formatted.
     * @throws ConstraintParserException If the constraint is not correctly formatted.
     */
    public static void extendModel(File sourceDimacsFile, String constraint, File destinationDimacsFile)
        throws FileNotFoundException, IOException, MalformedFileException, ConstraintParserException {
        
        // Read existing model
        DimacsReader reader = null;
        Project model = null;
        try {
        	reader = new DimacsReader(sourceDimacsFile);
            model = reader.getModel();
        } finally {
            if (null != reader) {
                reader.close();
            }
        }
        if (null != model) {
        	
        	// Cache Declarations + Constraints
        	DeclarationFinder varFinder = new DeclarationFinder(model, FilterType.ALL, null);
        	List<AbstractVariable> variables = varFinder.getVariableDeclarations(VisibilityType.ALL);
        	ConstraintFinder constraintFinder = new ConstraintFinder(model);
        	List<Constraint> constraints = constraintFinder.getConstraints();
        	
        	// Copy Model
        	Project clonedModel = cloneProject(variables,constraints);
        			
        	// Test created model
        	if(DEBUG)
        	{
        		System.out.println("Original Model");
        		System.out.println("variables = "+variables.size());
        		System.out.println("constraints = "+constraints.size());
//        		StringWriter sWriter = new StringWriter();
//        		IVMLWriter iWriter = new IVMLWriter(sWriter);
//        		clonedModel.accept(iWriter);
//        		System.out.println(sWriter);
        	}
        	
            // Extend model
            ModelExtender extender = new ModelExtender(clonedModel);
            extender.addConstraint(constraint);
            
//             Saved extended model

            if(DEBUG) {
            	System.out.println("Extended Model");
            	varFinder = new DeclarationFinder(clonedModel, FilterType.ALL, null);
            	variables = varFinder.getVariableDeclarations(VisibilityType.ALL);
            	constraintFinder = new ConstraintFinder(clonedModel);
            	constraints = constraintFinder.getConstraints();
            	System.out.println("variables = "+variables.size());
        		System.out.println("constraints = "+constraints.size());
            	FileWriter fWriter = new FileWriter("res/dest.dimacs");
                DimacsWriter writer = new DimacsWriter(clonedModel, fWriter);
                writer.write();
                fWriter.close();
            } else {
            	FileWriter fWriter = new FileWriter(destinationDimacsFile);
                DimacsWriter writer = new DimacsWriter(clonedModel, fWriter);
                writer.write();
                fWriter.close();
            }
           
//            FileWriter fWriter = new FileWriter(destinationDimacsFile);
//            DimacsWriter writer = new DimacsWriter(clonedModel, fWriter);

            
        }
    }
    private static Project cloneProject(List<AbstractVariable> variables,
    		List<Constraint> constraints) {
    	Project clonedModel = new Project("clonedModel");
    	for (int i = 0; i < variables.size(); i++) {
    		clonedModel.add(variables.get(i));
    	}
    	for (int i = 0; i < constraints.size(); i++) {
    		clonedModel.add(constraints.get(i));
    	}
    	return clonedModel;
    }

	// this method add constraints to a model
    public static void extendModel(Project model, String constraint, File destinationDimacsFile)
            throws FileNotFoundException, IOException, MalformedFileException, ConstraintParserException {
            
//            // Read existing model
//            DimacsReader reader = null;
//            Project model = null;
//            try {
//                reader = new DimacsReader(sourceDimacsFile);
//                model = reader.getModel();
//            } finally {
//                if (null != reader) {
//                    reader.close();
//                }
//            }
            if (null != model) {
            	
            	// Cache Declarations + Constraints
            	DeclarationFinder varFinder = new DeclarationFinder(model, FilterType.ALL, null);
            	List<AbstractVariable> variables = varFinder.getVariableDeclarations(VisibilityType.ALL);
            	ConstraintFinder constraintFinder = new ConstraintFinder(model);
            	List<Constraint> constraints = constraintFinder.getConstraints();
            	
            	// Copy Model
            	Project clonedModel = cloneProject(variables,constraints);
            	// Test created model
            	if(DEBUG)
            	{
            		StringWriter sWriter = new StringWriter();
            		IVMLWriter iWriter = new IVMLWriter(sWriter);
            		clonedModel.accept(iWriter);
            		System.out.println(sWriter);
            	}
            	
                // Extend model
                ModelExtender extender = new ModelExtender(clonedModel);
                extender.addConstraint(constraint);
                
                // Saved extended model
                FileWriter fWriter = new FileWriter(destinationDimacsFile);
                DimacsWriter writer = new DimacsWriter(clonedModel, fWriter);
                writer.write();
                fWriter.close();
            }
        }
    /**
     * Will extends the <tt>Project Model</tt> with the given <tt>constraint</tt> and return the
     * <tt>extendedModel</tt>.
     * @param model contains the model and must not be <tt>null</tt>.
     * @param constraint The constraint to append to the already existing model.
     * @throws MalformedFileException If the <tt>sourceDimacsFile</tt> is not correctly formatted.
     * @throws ConstraintParserException If the constraint is not correctly formatted.
     */
    public static Project extendModel(Project model, String constraint)
            throws MalformedFileException, ConstraintParserException {
            
            if (null != model) {
            	
            	// Cache Declarations + Constraints
            	DeclarationFinder varFinder = new DeclarationFinder(model, FilterType.ALL, null);
            	List<AbstractVariable> variables = varFinder.getVariableDeclarations(VisibilityType.ALL);
            	ConstraintFinder constraintFinder = new ConstraintFinder(model);
            	List<Constraint> constraints = constraintFinder.getConstraints();
            	
            	// Copy Model
            	Project clonedModel = cloneProject(variables,constraints);
            	
            	
            	// Test created model
            	if(DEBUG)
            	{
            		System.out.println("Original Model");
            		System.out.println("variables = "+variables.size());
            		System.out.println("constraints = "+constraints.size());
            		StringWriter sWriter = new StringWriter();
            		IVMLWriter iWriter = new IVMLWriter(sWriter);
            		clonedModel.accept(iWriter);
            		System.out.println(sWriter);
            	}
            	
            	// deintialized the variables after using them
            	varFinder = null;
            	variables = null;
            	constraintFinder = null;
            	constraints = null;
            	
                // Extend model
                ModelExtender extender = new ModelExtender(clonedModel);
                extender.addConstraint(constraint);
                if (DEBUG) {
                	System.out.println("Extended Model");
                	varFinder = new DeclarationFinder(clonedModel, FilterType.ALL, null);
                	variables = varFinder.getVariableDeclarations(VisibilityType.ALL);
                	constraintFinder = new ConstraintFinder(clonedModel);
                	constraints = constraintFinder.getConstraints();
                	System.out.println("variables = "+variables.size());
            		System.out.println("constraints = "+constraints.size());
                }
                return clonedModel;
            } else {
            	return null;
            }
			
        }
    
    
    /**
     * Main-Method to run the ModelExtender.
     * 
     * @param args Command line parameters:
     * <ol>
     *   <li>Path to .dimacs file</li>
     *   <li>Additional constraint</li>
     *   <li>Path to destination .dimacs file</li>
     * </ol>
     * @throws ConstraintParserException 
     * @throws MalformedFileException 
     * @throws IOException 
     * @throws FileNotFoundException 
     */
	public static void main(String[] args) throws FileNotFoundException, IOException, MalformedFileException, ConstraintParserException {
    	if(DEBUG) {
    		System.out.println("Calling extend math which returns a model as an object");
    		 File readModelFile = new File(args[0]);
    	        DimacsReader reader = null;
    	        try {
    	            reader = new DimacsReader(readModelFile);
    	        } catch (FileNotFoundException e) {
    	            System.err.println("File " + readModelFile.getPath() + " can't be opened for reading: "
    	                    + e.getMessage());
    	            System.exit(1);
    	        }
    	        Project model = null;
    	        try {
    	            model = reader.getModel();
    	        } catch (IOException | MalformedFileException e) {
    	            e.printStackTrace(System.err);
    	            System.exit(1);
    	        }
    	        
    	        try {
    	            reader.close();
    	        } catch (IOException e) {
    	            e.printStackTrace(System.err);
    	            System.exit(1);
    	        }
    	        if (model != null) {
    	        	
    	        	model = extendModel(model, "!2||3");
    	        	DeclarationFinder varFinder = new DeclarationFinder(model, FilterType.ALL, null);
                	List<AbstractVariable> variables = varFinder.getVariableDeclarations(VisibilityType.ALL);
                	ConstraintFinder constraintFinder = new ConstraintFinder(model);
                	List<Constraint> constraints = constraintFinder.getConstraints();
                	System.out.println("Variables and Constraint from model returned by extendModel method");
                	System.out.println("variables = "+variables.size());
            		System.out.println("constraints = "+constraints.size());
            		varFinder = null;
    	        	variables = null;
    	        	constraintFinder = null;
    	        	constraints = null;
    	        	getClauses(model);
    	        }
    	}
    	else {
        if (args.length < 3) {
            System.err.println("Not enough command line arguments");
            System.err.println("Usage: path/to/in/model.dimacs \"a && b<7\" path/to/out/model.dimacs");
            System.exit(1);
        }
        
        File readModelFile = new File(args[0]);
        DimacsReader reader = null;
        try {
            reader = new DimacsReader(readModelFile);
        } catch (FileNotFoundException e) {
            System.err.println("File " + readModelFile.getPath() + " can't be opened for reading: "
                    + e.getMessage());
            System.exit(1);
        }
        
        Project model = null;
        try {
            model = reader.getModel();
        } catch (IOException | MalformedFileException e) {
            e.printStackTrace(System.err);
            System.exit(1);
        }
        
        try {
            reader.close();
        } catch (IOException e) {
            e.printStackTrace(System.err);
            System.exit(1);
        }
        
        ModelExtender extender = new ModelExtender(model);
        try {
            extender.addConstraint(args[1]);
            
        } catch (ConstraintParserException e) {
            e.printStackTrace(System.err);
        }
        
        File writeModelFile = new File(args[2]);
        try {
        	DimacsWriter writer = new DimacsWriter(model, new FileWriter(writeModelFile));
            writer.write();
        } catch (IOException e) {
            e.printStackTrace(System.err);
        }
    }
    }
	private static void getClauses(Project model) {
		// TODO Auto-generated method stub
		 DeclarationFinder declarationFinder = new DeclarationFinder(model, FilterType.ALL, null);
	        List<AbstractVariable> variableDeclarations = declarationFinder.getVariableDeclarations(VisibilityType.ALL);
	        Collections.sort(variableDeclarations, new Comparator<AbstractVariable>() {
	            public int compare(AbstractVariable o1, AbstractVariable o2) {
	                return o1.getName().compareTo(o2.getName());
	            }
	        });
	        
	        // Get all Constraints
	        ConstraintFinder constraintFinder = new ConstraintFinder(model);
	        List<Constraint> constraints = constraintFinder.getConstraints();
	        
	        // Write variables
	        HashMap<String,Integer> clauseVariables = new HashMap<String, Integer>(); 
	        Integer i = 1;
	        for (AbstractVariable variable : variableDeclarations) {
	            clauseVariables.put(variable.getName(), i);
	            i++;
	        }
	        i = 1;
	       HashMap<Integer, List<Integer>> allClauses= new HashMap<Integer, List<Integer>>();
	        
	        for (Constraint constraint : constraints) {
	          
	        	String[] cons = constraint.toString().split("=");
	            allClauses.put(i, getClause(cons[1].trim(),clauseVariables));
	            i++;
	        }
	}
	
	private static List<Integer> getClause(String cls, HashMap<String, Integer> clauseVariables) {
		String[] subCls = cls.split(" ");
		final String NOT = "not";
		final String OR = "or";
		boolean isNot = false;
		List<Integer> cnf = new ArrayList<Integer>();
		for (int i=0; i < subCls.length; i++) {
			String ss = subCls[i];
			if(ss.equals(NOT)) {
				isNot = true;
			} else if (ss.equals(OR)) {
				// ignore or operation
			} else {
				if(clauseVariables.containsKey(ss)) {
					Integer number = clauseVariables.get(ss);
					if(isNot) {
						number = -number;
						isNot = false;
					} 
					cnf.add(number);

				} else {
					throw new RuntimeException();
				}

			}
		}
		return cnf;
	}
	
}
