package de.uni_hildesheim.sse.trans.convert;

import java.util.List;

import de.uni_hildesheim.sse.trans.Main;
import net.ssehub.easy.basics.logger.EASyLoggerFactory;
import net.ssehub.easy.basics.logger.EASyLoggerFactory.EASyLogger;
import net.ssehub.easy.varModel.model.AbstractVariable;
import net.ssehub.easy.varModel.model.Constraint;
import net.ssehub.easy.varModel.model.Project;
import net.ssehub.easy.varModel.model.filter.ConstraintFinder;
import net.ssehub.easy.varModel.model.filter.DeclarationFinder;
import net.ssehub.easy.varModel.model.filter.DeclarationFinder.VisibilityType;
import net.ssehub.easy.varModel.model.filter.FilterType;

/**
 * Converts a Boolean {@link Project} into CNF form.
 * @author El-Sharkawy
 *
 */
public class CNFConverter {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(CNFConverter.class, Main.ID);
    
    /**
     * Converts a Boolean {@link Project} into CNF form.
     * @param project A Boolean {@link Project} containing only constraints with AND, OR, and NOT operations.
     * @param converter A Strategy which is able to convert boolean formulas into CNF form.
     */
    public static void convert(Project project, ICNFConvertStrategy converter) {
        // Store complete content of project (imports are not used in this project).
        DeclarationFinder declfinder = new DeclarationFinder(project, FilterType.ALL, null);
        ConstraintFinder constFinder = new ConstraintFinder(project);
        
        // Clear old content
        project.clear();
        
        // 1. Re-add Declarations
        List<AbstractVariable> declarations = declfinder.getVariableDeclarations(VisibilityType.ALL);
        for (int i = 0, n = declarations.size(); i < n; i++) {
            project.add(declarations.get(i));
        }
        declfinder = null;
        declarations = null;
        
        // 2. Convert the constraints
        List<Constraint> constraints = constFinder.getConstraints();
        
        int lastNum = 0;
        long ms = 0;
        long timeLast = System.currentTimeMillis();
        
        for (int i = 0, n = constraints.size(); i < n; i++) {
            Constraint constraint = constraints.get(i);
            DisjunctionChecker checker = new DisjunctionChecker(constraint);
            if (checker.isDisjunctionTerm()) {
                project.add(constraint);
            } else {
                converter.convert(constraint);
            }
            
            ms = System.currentTimeMillis() - timeLast;
            if (ms > 5000) {
                timeLast = System.currentTimeMillis();
                LOGGER.info("Progress: " + i + "/" + constraints.size()
                        + " (" + (i - lastNum) + " in last 5 seconds)");
                lastNum = i;
            }
            
            /*
             *  Release constraint to deallocate memory.
             *  set(index, null) avoids Array.copy
             */
            constraints.set(i, null);
        }
    }
    
}
