package de.uni_hildesheim.sse.trans.convert;

import net.ssehub.easy.varModel.cst.AttributeVariable;
import net.ssehub.easy.varModel.cst.Comment;
import net.ssehub.easy.varModel.cst.CompoundAccess;
import net.ssehub.easy.varModel.cst.CompoundInitializer;
import net.ssehub.easy.varModel.cst.ConstantValue;
import net.ssehub.easy.varModel.cst.ContainerInitializer;
import net.ssehub.easy.varModel.cst.ContainerOperationCall;
import net.ssehub.easy.varModel.cst.IConstraintTreeVisitor;
import net.ssehub.easy.varModel.cst.IfThen;
import net.ssehub.easy.varModel.cst.Let;
import net.ssehub.easy.varModel.cst.OCLFeatureCall;
import net.ssehub.easy.varModel.cst.Parenthesis;
import net.ssehub.easy.varModel.cst.Self;
import net.ssehub.easy.varModel.cst.UnresolvedExpression;
import net.ssehub.easy.varModel.cst.Variable;
import net.ssehub.easy.varModel.model.Constraint;
import net.ssehub.easy.varModel.model.datatypes.OclKeyWords;

/**
 * Checks whether a given Constraint is a disjunction.
 * 
 * @author El-Sharkawy
 * 
 */
class DisjunctionChecker implements IConstraintTreeVisitor {

    private boolean isDisjunctionTerm;

    /**
     * Sole constructor for this class,
     * will check whether the given constraint is a disjunction term.
     * @param constraint The constraint to check. This constraint can only contain AND, OR, and NOT operations,
     *     otherwise this checker will not work.
     * @see #isDisjunctionTerm()
     */
    DisjunctionChecker(Constraint constraint) {
        isDisjunctionTerm = true;
        constraint.getConsSyntax().accept(this);
    }
    
    /**
     * Returns whether the given constraint is a disjunction term.
     * @return <tt>true</tt> if the given constraint is a disjunction term,
     * <tt>false</tt> otherwise.
     */
    boolean isDisjunctionTerm() {
        return isDisjunctionTerm;
    }

    @Override
    public void visitConstantValue(ConstantValue value) {
        // Nothing to do, end here
    }

    @Override
    public void visitVariable(Variable variable) {
        // Nothing to do, end here
    }
    
    @Override
    public void visitAnnotationVariable(AttributeVariable variable) {
        visitVariable(variable);
    }

    @Override
    public void visitParenthesis(Parenthesis parenthesis) {
        parenthesis.getExpr().accept(this);
    }

    @Override
    public void visitComment(Comment comment) {
        // Nothing to do, end here
    }

    @Override
    public void visitOclFeatureCall(OCLFeatureCall call) {
        String operation = call.getOperation();
        if (OclKeyWords.OR.equals(operation) || OclKeyWords.NOT.equals(operation)) {
            call.getOperand().accept(this);
            for (int i = 0, n = call.getParameterCount(); i < n; i++) {
                call.getParameter(i).accept(this);
            }
        } else {
            isDisjunctionTerm = false;
        }
    }

    @Override
    public void visitLet(Let let) {
        // Not supported by this Translator
        isDisjunctionTerm = false;
    }

    @Override
    public void visitIfThen(IfThen ifThen) {
        // Not supported by this Translator
        isDisjunctionTerm = false;
    }

    @Override
    public void visitContainerOperationCall(ContainerOperationCall call) {
        // Not supported by this Translator
        isDisjunctionTerm = false;
    }

    @Override
    public void visitCompoundAccess(CompoundAccess access) {
        // Not supported by this Translator
        isDisjunctionTerm = false;
    }

    @Override
    public void visitUnresolvedExpression(UnresolvedExpression expression) {
        // Not supported by this Translator
        isDisjunctionTerm = false;
    }

    @Override
    public void visitCompoundInitializer(CompoundInitializer initializer) {
         // Not supported by this Translator
        isDisjunctionTerm = false;
    }

    @Override
    public void visitContainerInitializer(ContainerInitializer initializer) {
        // Not supported by this Translator
        isDisjunctionTerm = false;
    }
    
    @Override
    public void visitSelf(Self self) {
        // Not supported by this Translator
        isDisjunctionTerm = false;
    }

}
