package de.uni_hildesheim.sse.trans.cli;

import java.io.File;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import de.uni_hildesheim.sse.model.varModel.IvmlKeyWords;
import de.uni_hildesheim.sse.utils.logger.AdvancedJavaLogger;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.LoggingLevel;

/**
 * Parser for parsing the command line arguments.
 * @author El-Sharkawy
 *
 */
@SuppressWarnings("static-access")
public class ArgumentsParser {

    private static final Options OPTIONS = new Options();
    
    private static final String CMD_IN = "in";
    private static final String CMD_OUT = "out";
    private static final String CMD_COMMENT = "comment";
    private static final String CMD_HELP = "h";
    private static final String CMD_VERSION = "v";
    
    static {
        createOption("inputFile", "The input file which shall be translated", CMD_IN, true);
        createOption("outputFile", "The output file, where the translated model shall be saved", CMD_OUT, true);
        createOption(CMD_COMMENT, "An optional comment, which shall be added to the translated result", CMD_COMMENT,
            false);
        OPTIONS.addOption(CMD_HELP, "help", false, "Prints this help isntead of perfoming the model translation");
        OPTIONS.addOption(CMD_VERSION, "version", true, "Optional: The version of the source model (in IVML syntax)");
        
        EASyLoggerFactory.INSTANCE.setLogger(new AdvancedJavaLogger());
        EASyLoggerFactory.INSTANCE.setLoggingLevel(LoggingLevel.INFO);
    }
    
   /**
    * Creates and option with an argument.
    * @param argName The command which shall be passed to the passed to this application, e.g -in
    * @param description A description for the user.
    * @param cmd The unique identifier of the option.
    * @param required Specification whether the parameter must be passed to this program, if <tt>false</tt> it is only
    *     optional.
    */
    private static void createOption(String argName, String description, String cmd, boolean required) {
        Option option = OptionBuilder.withArgName(argName).hasArg().withDescription(description).create(cmd);
        
        if (required) {
            option.isRequired();
        }
        
        OPTIONS.addOption(option);
    }
    
    /**
     * Parses the command line arguments passed to this program.
     * @param args The command line arguments passed to this program.
     * @return The parsed arguments.
     */
    public static Arguments parse(String[] args) {
        CommandLineParser parser = new BasicParser();
        
        Arguments arguments = new Arguments();
        
        if (args != null && args.length >= 2) {
            arguments.setHelp(false);
        }
        
        try {
            CommandLine cmd = parser.parse(OPTIONS, args);
            if (cmd.hasOption(CMD_IN)) {
                arguments.setInputFile(new File(cmd.getOptionValue(CMD_IN)));
            } else {
                arguments.setHelp(true);
            }
            if (cmd.hasOption(CMD_OUT)) {
                arguments.setOutputFile(new File(cmd.getOptionValue(CMD_OUT)));
            } else {
                arguments.setHelp(true);
            }
            if (cmd.hasOption(CMD_COMMENT)) {
                arguments.setComment(cmd.getOptionValue(CMD_COMMENT) + IvmlKeyWords.LINEFEED);
            }
            if (cmd.hasOption(CMD_VERSION)) {
                arguments.setVersion(cmd.getOptionValue(CMD_VERSION));
            }
            if (cmd.hasOption(CMD_HELP)) {
                arguments.setHelp(true);
            }
        } catch (ParseException exp) {
            arguments.setHelp(true);
            System.err.println("Parsing failed. Reason: " + exp.getMessage());

        }
        
        return arguments;
    }
    
    /**
     * Prints out the command line options.
     */
    public static void printHelp() {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("ModelTranslator", OPTIONS);
    }
}
