package de.uni_hildesheim.sse.trans.in.rsf;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;

import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.BooleanType;
import de.uni_hildesheim.sse.model.varModel.datatypes.Enum;
import de.uni_hildesheim.sse.model.varModel.datatypes.IDatatype;
import de.uni_hildesheim.sse.model.varModel.datatypes.IntegerType;
import de.uni_hildesheim.sse.model.varModel.datatypes.RealType;
import de.uni_hildesheim.sse.model.varModel.datatypes.StringType;
import de.uni_hildesheim.sse.trans.in.AbstractReader;
import de.uni_hildesheim.sse.trans.in.ParserException;
import de.uni_hildesheim.sse.trans.in.ParserException.ParserExceptionType;

/**
 * This reader reads <tt>*.rsf</tt> files, as they are created by the patched version of
 * <a href="https://github.com/ckaestne/undertaker">Undertaker's Dumpconf</a> tool.
 * These models must be pure boolean already.
 * @author El-Sharkawy
 *
 */
public class RSFReader extends AbstractReader {
    
    private Map<String, RSFItem> items;
    
    /**
     * Loads a <a href="https://github.com/ckaestne/undertaker">Undertaker's Dumpconf</a> rsf file.
     * @param rsfFile A RSF file, created by Undertaker's dumpconf
     * @throws FileNotFoundException if the file does not exist, is a directory rather than a regular file,
     * or for some other reason cannot be opened for reading.
     */
    public RSFReader(File rsfFile) throws FileNotFoundException {
        super(rsfFile);
        items = new LinkedHashMap<String, RSFItem>();
    }
    
    /**
     * Returns the specifies {@link RSFItem}, for the given name.
     * @param name The name of the KConfig item.
     * @return The {@link RSFItem} representing the KConfig item.
     */
    private RSFItem getItem(String name) {
        RSFItem item = items.get(name);
        if (null == item) {
            item = new RSFItem(name);
            items.put(name, item);
        }
        
        return item;
    }

    @Override
    public Project read() throws IOException, ParserException {
        readItems();
        Project model = getModel();
        // 1. Create Tristate type
        Enum tristate = new Enum("Tristate", model, "Undefined", "Defined", "Module");
        model.add(tristate);
        
        // 2. Add all Variables with correct datatype
        for (RSFItem item : items.values()) {
            IDatatype type = null;
            System.out.println(item.getName());
            switch(item.getDatatype()) {
            case BOOLEAN:
                type = BooleanType.TYPE;
                break;
            case TRISTATE:
                type = tristate;
                break;
            case INTEGER:
                type = IntegerType.TYPE;
                break;
            case HEX:
                // TODO handle as enum
                type = RealType.TYPE;
                break;
            case STRING:
                type = StringType.TYPE;
                break;
            default:
                throw new ParserException(ParserExceptionType.NOT_SUPPORTED_DATATYPE);
            }
            getVariable(item.getName(), type);
        }
        
        return model;
    }

    /**
     * First step of the translation, reads the items of the RSF file.
     * @throws IOException If an I/O error occurs
     * @throws ParserException If the input file contains a field, which is not supported by the Reader.
     */
    private void readItems() throws IOException, ParserException {
        BufferedReader reader = getReader();
        String line;
        while ((line = reader.readLine()) != null) {
            // Skip lines starting with a #
            if (line.charAt(0) != '#') {
                String[] columns = line.split("\t");
                String fieldType = columns[0];
                String itemName = columns[1];
                //System.out.println(itemName);
                RSFItem item = getItem(itemName);
                switch (fieldType) {
                case "Item":
                    item.setDatatype(columns[2]);
                    // Datatype in [2]
                    break;
                case "Prompt":
                    // Comment in [2]
                    break;
                case "Default":
                    // Default value in [3], condition [2]
                    break;
                case "HasPrompts":
                    break;
                case "ItemSelects":
                    break;
                case "Depends":
                    break;
                case "Choice":
                    // TODO Check whether this is correct
                    item.setDatatype("boolean");
                    break;
                case "ChoiceItem":
                    // related choice in [2]
                    break;
                case "Range":
                    // Range in [2], condition in [3]
                    break;
                default:
                    throw new ParserException(ParserExceptionType.NOT_SUPPORTED_FIELD);
                }
            }
        }
    }
}
