package de.uni_hildesheim.sse.trans.convert;

import de.uni_hildesheim.sse.model.confModel.AssignmentState;
import de.uni_hildesheim.sse.model.confModel.Configuration;
import de.uni_hildesheim.sse.model.confModel.ConfigurationException;
import de.uni_hildesheim.sse.model.confModel.IDecisionVariable;
import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cstEvaluation.EvaluationVisitor;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.values.BooleanValue;
import de.uni_hildesheim.sse.trans.Bundle;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Alternative implementation of {@link MaxTermConverter}, which uses a backtracking algorithm
 * instead of iteration to find all max terms.
 * 
 * @author Adam Krafczyk
 */
public class MaxTermConverter2 extends MaxTermConverter {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(MaxTermConverter2.class, Bundle.ID);


    @Override
    protected void handleSmallConstraint(ConstraintSyntaxTree originalConstraint, AbstractVariable[] declarationArray,
        Configuration config) {
        
        handleConstraint(originalConstraint, declarationArray, config, 0);
    }
    
    @Override
    protected void handleBigConstraint(ConstraintSyntaxTree originalConstraint, AbstractVariable[] declarationArray,
        Configuration config) {
        
        handleConstraint(originalConstraint, declarationArray, config, 0);
    }
    
    /**
     * Converts a constraint into a CNF formula.
     * @param originalConstraint The constraint which shall be converted into CNF formula.
     *     this constraint may consists only of AND, OR, and NOT operations.
     * @param declarationArray The {@link AbstractVariable}s, which are used inside the <tt>originalConstraint</tt>.
     * @param config A temporary {@link Configuration} containing only <tt>originalConstraint</tt>
     *     and <tt>declarationArray</tt>
     * @param var the variable to change
     */
    protected void handleConstraint(ConstraintSyntaxTree originalConstraint, AbstractVariable[] declarationArray,
            Configuration config, int var) {
        
        if (var >= declarationArray.length) {
            return;
        }
        
        try {
            IDecisionVariable decisionVar = config.getDecision(declarationArray[var]);
            
            decisionVar.setValue(BooleanValue.TRUE, AssignmentState.ASSIGNED);
            int result = checkConstraint(originalConstraint, config);
            if (result == 0) {
                addExpression(declarationArray, var, config);
            } else if (result == -1) {
                handleConstraint(originalConstraint, declarationArray, config, var + 1);
            }
            
            decisionVar.setValue(BooleanValue.FALSE, AssignmentState.ASSIGNED);
            result = checkConstraint(originalConstraint, config);
            if (result == 0) {
                addExpression(declarationArray, var, config);
            } else if (result == -1) {
                handleConstraint(originalConstraint, declarationArray, config, var + 1);
            }
            
            decisionVar.setValue(null, AssignmentState.UNDEFINED);
            
        } catch (ConfigurationException e) {
            LOGGER.exception(e);
        }
    }
    
    /**
     * Checks whether the constraint is already true, false or still undefined with the given variables set.
     * 
     * @param constraint The {@link ConstraintSyntaxTree} that represents the boolean expression
     * @param config the {@link Configuration} that holds the variable states
     * 
     * @return 0 if the expression is false; 1 if the expression is true; -1 if the expression is undefined
     */
    private int checkConstraint(ConstraintSyntaxTree constraint, Configuration config) {
        EvaluationVisitor evaluationVisitor = new EvaluationVisitor(config, null, false, null);
        constraint.accept(evaluationVisitor);
        
        int result = -1;
        if (evaluationVisitor.constraintFailed()) {
            result = 0;
        } else if (evaluationVisitor.constraintFulfilled()) {
            result = 1;
        }
        return result;
    }
    
    /**
     * Adds the boolean expression to the model via {@link MaxTermConverter#addConstraint(ConstraintSyntaxTree)}.
     * 
     * @param declarationArray Array with all variable declarations
     * @param var The number of variables that is defined (0 to var -1 in the declaration array)
     * @param config The {@link Configuration} that holds the states of the variables
     */
    private void addExpression(AbstractVariable[] declarationArray, int var, Configuration config) {
        boolean[] state = new boolean[var + 1];
        for (int i = 0; i < state.length; i++) {
            state[i] = ((BooleanValue) config.getDecision(declarationArray[i]).getValue()) == BooleanValue.TRUE;
        }
        addConstraint(createNegatedORExpression(declarationArray, state));
    }
    
}
