package de.uni_hildesheim.sse.trans.convert;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Parenthesis;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.model.varModel.filter.ConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder.VisibilityType;
import de.uni_hildesheim.sse.model.varModel.filter.DeclrationInConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.filter.FilterType;

/**
 * Optimizes a model in CNF in various ways.
 * 
 * @author Adam Krafczyk
 */
public class ModelOptimizer {

    private Project model;
    
    /**
     * Creates a {@link ModelOptimizer} for the given project.
     * @param model The project to be checked by this checker
     */
    public ModelOptimizer(Project model) {
        this.model = model;
    }
    
    /**
     * Removes all unused variables (i.e. variables that are not used in constraints) from the project.
     * 
     * @return The number of removed variables.
     */
    public int removeUnusedVariables() {
        ConstraintFinder cFinder = new ConstraintFinder(model);
        DeclarationFinder varFinder = new DeclarationFinder(model, FilterType.ALL, null);
        
        List<AbstractVariable> varList = varFinder.getVariableDeclarations(VisibilityType.ALL);
        Set<AbstractVariable> varSet = new HashSet<AbstractVariable>(varList);
        
        // 1) Go through all constraints and remove used variables from the set
        List<Constraint> constraints = cFinder.getConstraints();
        for (int i = 0, n = constraints.size(); i < n; i++) {
            Constraint constraint = constraints.get(i);
            DeclrationInConstraintFinder finder = new DeclrationInConstraintFinder(constraint.getConsSyntax());
            Set<AbstractVariable> declarations = finder.getDeclarations();
            varSet.removeAll(declarations);
            
            /*
             *  Release constraint to deallocate memory.
             *  set(index, null) avoids Array.copy
             */
            constraints.set(i, null);
        }
        
        // 2) All remaining variables in the set are unused
        for (AbstractVariable var : varSet) {
            model.removeElement(var);
        }
        
        return varSet.size();
    }
    
    /**
     * Check for variables that always have to be true or false and update all other constraints accordingly.
     * 
     * @return The number of removed constraints.
     */
    public int handleConstantVariables() {
        Set<AbstractVariable> alwaysTrueVariables = new HashSet<AbstractVariable>();
        Set<AbstractVariable> alwaysFalseVariables = new HashSet<AbstractVariable>();
        
        ConstraintFinder cFinder = new ConstraintFinder(model);
        List<Constraint> constraints = cFinder.getConstraints();
        
        // 1) Go through all constraints and find variables that always have to be true or false
        for (int i = 0, n = constraints.size(); i < n; i++) {
            ConstraintSyntaxTree tree = constraints.get(i).getConsSyntax();
            
            // TODO: check if there is only 1 variable and use EvaluationVistor instead?
            
            if (tree instanceof Variable) {
                Variable var = (Variable) tree;
                alwaysTrueVariables.add(var.getVariable());
            } else if (tree instanceof OCLFeatureCall) {
                OCLFeatureCall call = (OCLFeatureCall) tree;
                if (call.getOperation().equals(OclKeyWords.NOT) && call.getOperand() instanceof Variable) {
                    Variable var = (Variable) call.getOperand();
                    alwaysFalseVariables.add(var.getVariable());
                }
            }
        }
        
        // 2) Remove all constraints that contain a not negated alwaysTrueVariable or contain a negated
        //    alwaysFalseVariable.
        int numRemoved = 0;
        for (int i = 0, n = constraints.size(); i < n; i++) {
            Constraint constraint = constraints.get(i);
            ConstraintSyntaxTree tree = constraint.getConsSyntax();
            
            DeclrationInConstraintFinder declarationFinder = new DeclrationInConstraintFinder(tree);
            Set<AbstractVariable> declarations = declarationFinder.getDeclarations();
            
            // Skip constraints with only 1 variable since this are the constraints that define the constatn variables
            if (declarations.size() <= 1) {
                continue;
            }
            
            for (AbstractVariable var : declarations) {
                if (alwaysFalseVariables.contains(var)) {
                    if (isVariableNegated(tree, var)) {
                        tree = null;
                        break;
                    }
                    // TODO: we could say that the model is not solvable if the else branch is reached
                } else if (alwaysTrueVariables.contains(var)) {
                    if (!isVariableNegated(tree, var)) {
                        tree = null;
                        break;
                    }
                    // TODO: we could say that the model is not solvable if the else branch is reached
                }
            }
            
            if (tree == null) {
                model.removeElement(constraint);
                numRemoved++;
            }
            
            /*
             *  Release constraint to deallocate memory.
             *  set(index, null) avoids Array.copy
             */
            constraints.set(i, null);
        }
        
        return numRemoved;
    }
    
    /**
     * Returns whether the given CNF formula contains the variable negated or not. The tree must contain the variable.
     * @param tree The {@link ConstraintSyntaxTree} with the variable in CNF
     * @param variable The variable to search
     * @return Whether the variable is in a NOT call or not
     */
    private boolean isVariableNegated(ConstraintSyntaxTree tree, AbstractVariable variable) {
        return isVariableNegated(tree, variable, false);
    }
    
    /**
     * Returns whether the given CNF formula contains the variable negated or not. The tree must contain the variable.
     * @param tree The {@link ConstraintSyntaxTree} with the variable in CNF
     * @param variable The variable to search
     * @param negated If the previous recursion step is a NOT call
     * @return Whether the variable is in a NOT call or not
     */
    private boolean isVariableNegated(ConstraintSyntaxTree tree, AbstractVariable variable, boolean negated) {
        
        boolean result = false;
        
        if (tree instanceof OCLFeatureCall) {
            OCLFeatureCall call = (OCLFeatureCall) tree;
            if (call.getOperation().equals(OclKeyWords.NOT)) {
                result = isVariableNegated(call.getOperand(), variable, true);
            } else {
                if (containsVariable(call.getOperand(), variable)) {
                    result = isVariableNegated(call.getOperand(), variable, negated);
                } else if (containsVariable(call.getParameter(0), variable)) {
                    result = isVariableNegated(call.getParameter(0), variable, negated);
                } else {
                    // TODO
                    throw new RuntimeException();
                }
            }
        } else if (tree instanceof Variable) {
            Variable var = (Variable) tree;
            if (var.getVariable().equals(variable)) {
                result = negated;
            }
        } else if (tree instanceof Parenthesis) {
            Parenthesis parenthesis = (Parenthesis) tree;
            result = isVariableNegated(parenthesis.getExpr(), variable, negated);
        } else {
            // TODO
            System.out.println(tree);
            throw new RuntimeException();
        }
        
        return result;
    }
    
    /**
     * Checks whether the given {@link ConstraintSyntaxTree} contains the variable.
     * @param tree The {@link ConstraintSyntaxTree} to check
     * @param variable The variable to find
     * @return true if the {@link ConstraintSyntaxTree} contains the variable
     */
    private boolean containsVariable(ConstraintSyntaxTree tree, AbstractVariable variable) {
        DeclrationInConstraintFinder finder = new DeclrationInConstraintFinder(tree);
        return finder.getDeclarations().contains(variable);
    }
    
}
