package de.uni_hildesheim.sse.trans.scenario;

import java.io.File;

import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Test;

import de.uni_hildesheim.sse.trans.AllTests;
import de.uni_hildesheim.sse.trans.DimacsTestUtils;
import de.uni_hildesheim.sse.trans.convert.OptimizationParameter;

/**
 * Tests the translation of a RSF file into DIMACS format.
 * @author El-Sharkawy
 *
 */
public class RsfToDimacsTranslationTest {
    
    /**
     * Tests whether Strings are translated correctly.
     */
    @Test
    public void testEmptyStrings() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_emptyString.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();

        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int extraFirmware = DimacsTestUtils.getNumberOfVariable(result, "EXTRA_FIRMWARE");
        int extraFirmwareDir = DimacsTestUtils.getNumberOfVariable(result, "EXTRA_FIRMWARE_DIR");
        
        /*
         * Test whether following constraint is included:
         * NOT(EXTRA_FIRMWARE_DIR) OR EXTRA_FIRMWARE
         */
        Assert.assertTrue("Error: Expected Constraint not included.",
                DimacsTestUtils.containsConstraint(result, extraFirmware, -1 * extraFirmwareDir));
    }
    
    /**
     * Tests whether ItemSelect statements are translated correctly into CNF constraints.
     */
    @Test
    public void testItemSelectStatements() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_ItemSelects.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int iwc3200top = DimacsTestUtils.getNumberOfVariable(result, "IWMC3200TOP") * -1;
        int fwLoader = DimacsTestUtils.getNumberOfVariable(result, "FW_LOADER");
        int blubb = DimacsTestUtils.getNumberOfVariable(result, "BLUBB") * -1;
        
        /* 
         * Test whether following constraint is included:
         * Not(IWMC3200TOP AND BLUBB) or FW_LOADER
         * -->
         * Not(IWMC3200TOP) OR NOT(BLUBB) or FW_LOADER
         */
        Assert.assertTrue("Error: Expected Constraint not included.",
            DimacsTestUtils.containsConstraint(result, iwc3200top, blubb, fwLoader));
    }
    
    /**
     * Tests whether the combination of prompts and default values are considered. Tests:
     * <ul>
     * <li>Variables with default and no prompt will create a constraint for the default</li>
     * <li>Testing of defaults with conditions</li>
     * <li>Testing of prompts with conditions</li>
     * <li>Testing negation of string variables</li>
     * </ul>
     */
    @Test
    @Ignore("TODO")
    public void testDefaultsAndPrompts() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_DefaultAndPrompts.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        System.out.println(result);
        int arch32 = DimacsTestUtils.getNumberOfVariable(result, "ARCH=x86");
        int arch64 = DimacsTestUtils.getNumberOfVariable(result, "ARCH=x86_64");
        int bit = DimacsTestUtils.getNumberOfVariable(result, "64BIT");
        
        /*
         * Test whether following 3 constraints are included:
         * - ARCH='x86_64'
         * - ARCH='x86_64' implies 64BIT
         *   -> Not(ARCH='x86_64') or 64BIT
         * - ARCH='x86' implies (64BIT or !64BIT)
         *   -> tautology -> skip this constraint
         * - (!ARCH='x86_64' and !ARCH='x86') implies !64Bit (combination of prompt and default)
         *   -> ARCH='x86_64' or ARCH='x86' or !64BIT
         */
        Assert.assertTrue("Error: ARCH='x86_64' constant not included",
            DimacsTestUtils.containsConstraint(result, arch64));
        Assert.assertTrue("Error: 64Bit default not included",
            DimacsTestUtils.containsConstraint(result, -1 * arch64, bit));
        Assert.assertTrue("Error: 64Bit default and prompt not included",
            DimacsTestUtils.containsConstraint(result, arch32, arch64, -1 * bit));
    }
    
    /**
     * Tests whether choices are translated correctly into CNF constraints.
     */
    @Test
    public void testChoices() {
        File input = new File(AllTests.INPUT_FOLDER, "testModel_Choices.rsf");
        OptimizationParameter noOptimization = new OptimizationParameter();
        
        // Test precondition
        Assert.assertFalse(noOptimization.hasAtLeastOneOption());
        
        // Translation
        String result = DimacsTestUtils.loadModel(input, noOptimization, true);
        int choice12 = DimacsTestUtils.getNumberOfVariable(result, "CHOICE_12");
        int hz100 = DimacsTestUtils.getNumberOfVariable(result, "HZ_100");
        int hz250 = DimacsTestUtils.getNumberOfVariable(result, "HZ_250");
        int hz300 = DimacsTestUtils.getNumberOfVariable(result, "HZ_300");
        int hz1000 = DimacsTestUtils.getNumberOfVariable(result, "HZ_1000");
        
        /*
         * Test whether the following constraints are included:
         * NOT(CHOICE_12) OR NOT(HZ_100) OR NOT(HZ_250)
         * NOT(CHOICE_12) OR NOT(HZ_100) OR NOT(HZ_300)
         * NOT(CHOICE_12) OR NOT(HZ_100) OR NOT(HZ_1000)
         * (skip other combinations)
         * NOT(CHOICE_12) OR HZ_100 OR HZ_250 OR HZ_300 OR HZ_1000
         */
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, -1 * choice12, -1 * hz100, -1 * hz250));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, -1 * choice12, -1 * hz100, -1 * hz300));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, -1 * choice12, -1 * hz100, -1 * hz1000));
        Assert.assertTrue("Error: Missing choice constraint",
                DimacsTestUtils.containsConstraint(result, -1 * choice12, hz100, hz250, hz300, hz1000));
    }

}
