package de.uni_hildesheim.sse.trans.in;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.FilenameUtils;

import de.uni_hildesheim.sse.model.validation.IvmlIdentifierCheck;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.BooleanType;

/**
 * This reader reads <tt>*.model</tt> files, as they are specified by
 * <a href="http://vamos.informatik.uni-erlangen.de/trac/undertaker">Undertaker</a> and
 * <a href="https://github.com/ckaestne/kconfigreader">KConfigReader</a>.
 * These models must be pure boolean already.
 * @author El-Sharkawy
 *
 */
public class ModelReader {
    private BufferedReader reader;
    private Project model;
    private Map<String, DecisionVariableDeclaration> variables;
    
    public ModelReader(File modelFile) throws FileNotFoundException {
        reader = new BufferedReader(new FileReader(modelFile));
        
        
        String name = FilenameUtils.removeExtension(modelFile.getName());
        while (!IvmlIdentifierCheck.isValidIdentifier(name)) {
            // Replace whitespaces
            name = name.replaceAll("\\s", "_");
            // Replace "punctuation" signs, like !, #, <, ...
            name = name.replaceAll("\\p{Punct}", "_");
            // If name starts with a number, add _ at the beginning
            if (name.matches("^\\d")) {
                name = "_" + name;
            }
        }
        
        model = new Project(name);
        variables = new HashMap<String, DecisionVariableDeclaration>();
    }
    
    /**
     * Returns a {@link DecisionVariableDeclaration} for the specified name. If such a variables was not created yet,
     * a new variable will be created and added to the {@link #model}.
     * @param name The name of the desired variable.
     * @return A boolean {@link DecisionVariableDeclaration} with the specified name.
     */
    private DecisionVariableDeclaration getVariable(String name) {
        DecisionVariableDeclaration variable = variables.get(name);
        if (null == variable) {
            variable = new DecisionVariableDeclaration(name, BooleanType.TYPE, model);
            model.add(variable);
            variables.put(name, variable);
        }
        
        return variable;
    }
    
    /**
     * Parses the given file and converts it to a {@link Project}.
     * @return The parsed file as a {@link Project}.
     * @throws IOException If an I/O error occurs
     */
    public Project read() throws IOException {
        String line;
        while ((line = reader.readLine()) != null) {
            // Skip lines starting with a #
            if (line.charAt(0) != '#') {
                System.out.println(line);
            }
        }
        
        return model;
    }
}
