package de.uni_hildesheim.sse.trans.in.rsf;

import java.util.ArrayList;
import java.util.List;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.datatypes.Enum;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.in.AbstractReader;
import de.uni_hildesheim.sse.trans.in.ParserException;

/**
 * Represents a condition read from a "Depends" line in a .rsf file.
 * The format looks like this: <br />
 * <code>
 * Depends <i>variable</i> <i>condition</i>
 * </code> <br />
 * <i>variable</i> can only be true, if <i>condition</i> is true. <br />
 * {@link RSFDependsCondition#getPureBooleanConstraintSyntaxTree(AbstractReader, String, String)} will convert it into:
 * <br />
 * <code>
 * NOT <i>variable</i> OR <i>condition</i>
 * </code> <br />
 * If condition contains a tristate variable and <i>variable</i> is a tristate variable, the following condition will
 * be added: <br />
 * <code>
 * NOT <i>variable_MODULE</i> OR <i>condition_variable_MODULE</i>
 * </code> <br />
 * TODO: what if it contains several tristate variables?
 * 
 * @author Adam Krafczyk
 */
class RSFDependsCondition extends RSFCondition {
//    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(RSFDependsCondition.class,
//            "RSFDependsCondition");

    private RSFItem variable;
    private List<String> conditions;
    
    /**
     * Creates a condition with the given variable and condition as read from a "Depends" line in a .rsf file.
     * @param variable The name of the variable
     * @param condition The condition as read from the .rsf file
     */
    RSFDependsCondition(RSFItem variable, String condition) {
        this.variable = variable;
        conditions = new ArrayList<String>();
        addCondition(condition);
    }
    
    /**
     * Adds another condition with the same base variable.
     * @param condition The other condition as read from another Depends variable line.
     */
    void addCondition(String condition) {
        // TODO error handling if only one is true?
        if (condition.startsWith("\"") && condition.endsWith("\"")) {
            condition = condition.substring(1, condition.length() - 1);
        }
        conditions.add(condition);
    }
    
    @Override
    List<ConstraintSyntaxTree> toPureBooleanConstraintSyntaxTree(RSFReader reader) throws ParserException {
        
        List<ConstraintSyntaxTree> conditionTrees = new ArrayList<ConstraintSyntaxTree>();
        
        for (String conditionString : conditions) {
            conditionTrees.add(getPureBooleanConstraintSyntaxTree(reader, conditionString, variable.getName()));
        }
        
        List<ConstraintSyntaxTree> result  = new ArrayList<ConstraintSyntaxTree>();
        
        boolean hasAtLeastOneNotNull = false;
        for (ConstraintSyntaxTree tree : conditionTrees) {
            if (tree != null) {
                hasAtLeastOneNotNull = true;
                break;
            }
        }
        
        if (hasAtLeastOneNotNull) {
            
            if (variable.getDatatype().equals(Datatype.TRISTATE)) {
                
                ConstraintSyntaxTree varTree = getUnselectedVariable(variable, reader);
                
                DecisionVariableDeclaration moduleDecl = reader.getVariable(variable.getName() + "_MODULE");
                ConstraintSyntaxTree moduleTree = new OCLFeatureCall(varPool.obtainVariable(moduleDecl),
                        OclKeyWords.NOT);
                
                for (ConstraintSyntaxTree conditionTree : conditionTrees) {
                    if (conditionTree != null) {
                        varTree = new OCLFeatureCall(varTree, OclKeyWords.OR, conditionTree);
                        moduleTree = new OCLFeatureCall(moduleTree, OclKeyWords.OR, conditionTree);
                    }
                }
                
                result.add(varTree);
                result.add(moduleTree);
                
            } else {
                
                ConstraintSyntaxTree tree = getUnselectedVariable(variable, reader);
                
                for (ConstraintSyntaxTree conditionTree : conditionTrees) {
                    if (conditionTree != null) {
                        tree = new OCLFeatureCall(tree, OclKeyWords.OR, conditionTree);
                    }
                }
                
                result.add(tree);
            }
        }
        
        return result;
    }
    
    @Override
    ConstraintSyntaxTree toNotBooleanConstraintSyntaxTree(AbstractReader reader, Enum tristate) throws ParserException {
        // TODO implement
        return null;
    }
    
}
