package de.uni_hildesheim.sse.trans.in.rsf;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import de.uni_hildesheim.sse.model.cst.Variable;

/**
 * All information items of one KConfig variable, read out of a RSF file.
 * @author El-Sharkawy
 *
 */
class RSFItem {
    private String name;
    private Datatype type;
    private boolean hasPrompt;
    private String promptCondition;
    private RSFDefaultCondition defaultCondition;
    private List<Variable> comparisonVariables;
    
    /**
     * Sole constructor for this datatype.
     * @param name The name of the KConfig variable.O
     */
    RSFItem(String name) {
        this.name = name;
    }
   
    /**
     * Getter for the name of the {@link RSFItem}.
     * @return The name of this item.
     */
    public String getName() {
        return name;
    }
    
    /**
    * Setter for the {@link Datatype}.
    * @param type The Datatype of the item, maybe in small letters.
    */
    public void setDatatype(String type) {
        this.type = Datatype.getDatatype(type); 
    }

    /**
     * Getter for the Datatype of the variable.
     * @return The Datatype of the item.
     * @return
     */
    public Datatype getDatatype() {
        return type;
    }
    
    /**
     * Getter for the HasPrompt attribute of the variable.
     * @return Whether the variable is changeable by the user.
     */
    public boolean hasPrompt() {
        return hasPrompt;
    }
    
    /**
     * Setter for the HasPrompt attribute of the variable.
     * @param hasPrompt Whether the variable is changeable by the user.
     */
    public void setHasPrompt(boolean hasPrompt) {
        this.hasPrompt = hasPrompt;
    }
    
    /**
     * Getter for the default condition of this variable.
     * @return The default condition of this variable.
     */
    public RSFDefaultCondition getDefaultCondition() {
        return defaultCondition;
    }
    
    /**
     * Setter for the default condition of this variable.
     * @param defaultCondition The default condition for this variable.
     */
    public void setDefaultCondition(RSFDefaultCondition defaultCondition) {
        this.defaultCondition = defaultCondition;
    }
    
    /**
     * Getter for the condition that toggles the prompt for the variable.
     * @return The condition; <code>null</code> if not found in the .rsf file.
     */
    public String getPromptCondition() {
        return promptCondition;
    }
    
    /**
     * Getter for the condition that toggles the prompt for the variable.
     * @param promptCondition The condition.
     */
    public void setPromptCondition(String promptCondition) {
        this.promptCondition = promptCondition;
    }
    
    /**
     * Adds a created comparison variable found in a constraint.
     * @param variable The variable created from a comparison found in a constraint (e.g. STR=n)
     */
    public void addComparisonVariable(Variable variable) {
        if (comparisonVariables == null) {
            comparisonVariables = new ArrayList<Variable>();
        }
        comparisonVariables.add(variable);
    }
    
    /**
     * Returns a {@link List} of all variables created from comparisons found in expressions.
     * @return An unmodifiable {@link List} of variables
     */
    public List<Variable> getComparisonVariables() {
        List<Variable> result = null;
        if (comparisonVariables != null) {
            result = Collections.unmodifiableList(comparisonVariables);
        }
        return result;
    }
    
    @Override
    public String toString() {
        return type.name() + " " + name;
    }
}
