package de.uni_hildesheim.sse.trans.in.rsf;

import java.util.ArrayList;
import java.util.List;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.varModel.datatypes.Enum;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.in.AbstractReader;
import de.uni_hildesheim.sse.trans.in.ParserException;

/**
 * Represents a condition read from a "ItemSelects" line in a .rsf file. <br />
 * The format looks like this: <br />
 * <code>
 * ItemSelects <i>variable</i> <i>selectedVaraible</i> <i>condition</i>
 * </code> <br />
 * <i>selectedVaraible</i> has to be set to true, if <i>variable</i> and <i>condition</i> are true. <br />
 * {@link RSFItemSelectsCondition#getPureBooleanConstraintSyntaxTree(AbstractReader, String, String)} will convert it
 * to: <br />
 * <code>
 * NOT <i>variable</i> OR NOT <i>condition</i> OR <i>selectedVaraible</i>
 * </code> <br />
 * or, if condition is "y", to: <br />
 * <code>
 * NOT <i>variable</i> OR <i>selectedVaraible</i>
 * </code> <br />
 * @author Adam Krafczyk
 */
class RSFItemSelectsCondition extends RSFCondition {
    
    private RSFItem variable;
    private String selectedVariable;
    private String condition;
    
    /**
     * Creates a condition with the given variable and condition as read from a "ItemSelects" line in a .rsf file.
     * @param variable The name of the variable
     * @param selectedVaraible The name of the variable that is selected
     * @param condition The condition as read from the .rsf file
     */
    RSFItemSelectsCondition(RSFItem variable, String selectedVaraible, String condition) {
        this.variable = variable;
        this.selectedVariable = selectedVaraible;
        // TODO error handling if only one is true?
        if (this.selectedVariable.startsWith("\"") && this.selectedVariable.endsWith("\"")) {
            this.selectedVariable = this.selectedVariable.substring(1, this.selectedVariable.length() - 1);
        }
        // TODO error handling if only one is true?
        if (this.selectedVariable.startsWith("'") && this.selectedVariable.endsWith("'")) {
            this.selectedVariable = this.selectedVariable.substring(1, this.selectedVariable.length() - 1);
        }
        this.condition = condition;
        // TODO error handling if only one is true?
        if (this.condition.startsWith("\"") && this.condition.endsWith("\"")) {
            this.condition = this.condition.substring(1, this.condition.length() - 1);
        }
    }
    
    @Override
    List<ConstraintSyntaxTree> toPureBooleanConstraintSyntaxTree(RSFReader reader) throws ParserException {
        ConstraintSyntaxTree notVar = getUnselectedVariable(variable, reader);
        
        RSFItem selectedItem = reader.getItemNoCreate(selectedVariable);
        ConstraintSyntaxTree selectedVar = getSelectedVariable(selectedItem, reader);
        
        ConstraintSyntaxTree condition = getPureBooleanConstraintSyntaxTree(reader, this.condition, variable.getName());
        
        
        List<ConstraintSyntaxTree> trees  = new ArrayList<ConstraintSyntaxTree>();
        
        if (condition != null) {
            ConstraintSyntaxTree notCondition = new OCLFeatureCall(condition, OclKeyWords.NOT);
            ConstraintSyntaxTree result = new OCLFeatureCall(notCondition, OclKeyWords.OR, selectedVar);
            result = new OCLFeatureCall(notVar, OclKeyWords.OR, result);
            trees.add(result);
            
            if (variable.getDatatype().equals(Datatype.TRISTATE)) {
                ConstraintSyntaxTree varModule = varPool.obtainVariable(
                        reader.getVariable(variable.getName() + "_MODULE"));
                ConstraintSyntaxTree notVarModule = new OCLFeatureCall(varModule, OclKeyWords.NOT);
                ConstraintSyntaxTree selectedVarModule = varPool.obtainVariable(
                        reader.getVariable(selectedVariable + "_MODULE"));
                
                ConstraintSyntaxTree moduleResult = new OCLFeatureCall(notVarModule, OclKeyWords.OR, selectedVarModule);
                moduleResult = new OCLFeatureCall(moduleResult, OclKeyWords.OR, notCondition);
                trees.add(moduleResult);
            }
            
        } else {
            trees.add(new OCLFeatureCall(notVar, OclKeyWords.OR, selectedVar));
            
            if (variable.getDatatype().equals(Datatype.TRISTATE)) {
                ConstraintSyntaxTree varModule = varPool.obtainVariable(
                        reader.getVariable(variable.getName() + "_MODULE"));
                ConstraintSyntaxTree notVarModule = new OCLFeatureCall(varModule, OclKeyWords.NOT);
                ConstraintSyntaxTree selectedVarModule = varPool.obtainVariable(
                        reader.getVariable(selectedVariable + "_MODULE"));
                
                trees.add(new OCLFeatureCall(notVarModule, OclKeyWords.OR, selectedVarModule));
            }
        }
        
        return trees;
    }
    
    @Override
    ConstraintSyntaxTree toNotBooleanConstraintSyntaxTree(AbstractReader reader, Enum tristate) throws ParserException {
        // TODO implement
        return null;
    }
    
}
