package de.uni_hildesheim.sse.trans.in;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.FilenameUtils;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.validation.IvmlIdentifierCheck;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.DecisionVariableDeclaration;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.BooleanType;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.utils.logger.AdvancedJavaLogger;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * This reader reads <tt>*.model</tt> files, as they are specified by
 * <a href="http://vamos.informatik.uni-erlangen.de/trac/undertaker">Undertaker</a> and
 * <a href="https://github.com/ckaestne/kconfigreader">KConfigReader</a>.
 * These models must be pure boolean already.
 * @author El-Sharkawy
 *
 */
public class ModelReader {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(ModelReader.class, "ModelReader");
    
    static {
        EASyLoggerFactory.INSTANCE.setLogger(new AdvancedJavaLogger());
        //EASyLoggerFactory.INSTANCE.setLoggingLevel(LoggingLevel.DEBUG);
    }
    
    private BufferedReader reader;
    private Project model;
    private Map<String, DecisionVariableDeclaration> variables;
    
    public ModelReader(File modelFile) throws FileNotFoundException {
        reader = new BufferedReader(new FileReader(modelFile));
        
        
        String name = FilenameUtils.removeExtension(modelFile.getName());
        while (!IvmlIdentifierCheck.isValidIdentifier(name)) {
            // Replace whitespaces
            name = name.replaceAll("\\s", "_");
            // Replace "punctuation" signs, like !, #, <, ...
            name = name.replaceAll("\\p{Punct}", "_");
            // If name starts with a number, add _ at the beginning
            if (name.matches("^\\d")) {
                name = "_" + name;
            }
        }
        
        model = new Project(name);
        variables = new HashMap<String, DecisionVariableDeclaration>();
    }
    
    /**
     * Returns a {@link DecisionVariableDeclaration} for the specified name. If such a variables was not created yet,
     * a new variable will be created and added to the {@link #model}.
     * @param name The name of the desired variable.
     * @return A boolean {@link DecisionVariableDeclaration} with the specified name.
     */
    private DecisionVariableDeclaration getVariable(String name) {
        DecisionVariableDeclaration variable = variables.get(name);
        if (null == variable) {
            variable = new DecisionVariableDeclaration(name, BooleanType.TYPE, model);
            model.add(variable);
            variables.put(name, variable);
        }
        
        return variable;
    }
    
    /**
     * Reads a single line into a {@link ConstraintSyntaxTree}.
     * 
     * @param line the line to read
     * @return a {@link ConstraintSyntaxTree} representing the line
     * @throws TODO
     */
    public ConstraintSyntaxTree parseLine(String line, int ebene) throws Exception {
        StringBuffer debugLine = new StringBuffer();
        for (int i = 0; i < ebene; i++) {
            debugLine.append('\t');
        }
        debugLine.append(line);
       
        if (line.startsWith("def(") && line.indexOf(')') == line.lastIndexOf(')')) {
            String variableName = line.substring("def(".length(), line.length() - 1);
            LOGGER.debug(debugLine.toString());
            return new Variable(getVariable(variableName));
        }
        
        if (line.startsWith("!def(") && line.indexOf(')') == line.lastIndexOf(')')) {
            String variableName = line.substring("!def(".length(), line.length() - 1);
            LOGGER.debug(debugLine.toString());
            return new OCLFeatureCall(new Variable(getVariable(variableName)), OclKeyWords.NOT);
        }
        
        int highestOperandPos = line.length();
        int highestOperandLevel = Integer.MAX_VALUE;
        String highestOperand = "";
        int currentLevel = 0;
        for (int i = 0; i < line.length(); i++) {
            char c = line.charAt(i);
            switch (c) {
            case '(':
                currentLevel++;
                break;
            case ')':
                currentLevel--;
                break;
                
            case '&':
                if (highestOperandLevel > currentLevel) {
                    highestOperand = OclKeyWords.AND;
                    highestOperandPos = i;
                    highestOperandLevel = currentLevel;
                }
                break;
            
            case '|':
                if (highestOperandLevel > currentLevel) {
                    highestOperand = OclKeyWords.OR;
                    highestOperandPos = i;
                    highestOperandLevel = currentLevel;
                }
                break;
            
            default:
                break;
            }
        }
        
        debugLine.append(" -> " + highestOperand + " (" + highestOperandPos + ")");
        LOGGER.debug(debugLine.toString());
        
        StringBuffer left = new StringBuffer(line.substring(0, highestOperandPos));
        StringBuffer right = new StringBuffer(line.substring(highestOperandPos + 1, line.length()));
        
        currentLevel = 0;
        for (int i = 0; i < left.length(); i++) {
            char c = left.charAt(i);
            if ((c == '(' || c == ')') && currentLevel < highestOperandLevel) {
                left.replace(i, i + 1, "");
            }
            if (c == '(') {
                currentLevel++;
            }
            if (c == ')') {
                currentLevel--;
            }
        }
        currentLevel = 0;
        for (int i = right.length() - 1; i >= 0; i--) {
            char c = right.charAt(i);
            if ((c == '(' || c == ')') && currentLevel < highestOperandLevel) {
                right.replace(i, i + 1, "");
            }
            if (c == ')') {
                currentLevel++;
            }
            if (c == '(') {
                currentLevel--;
            }
        }
        
        ConstraintSyntaxTree leftSide = parseLine(left.toString(), ebene + 1);
        ConstraintSyntaxTree rightSide = parseLine(right.toString(), ebene + 1);
        
        return new OCLFeatureCall(leftSide, highestOperand, rightSide);
    }
    
    /**
     * Parses the given file and converts it to a {@link Project}.
     * @return The parsed file as a {@link Project}.
     * @throws IOException If an I/O error occurs
     */
    public Project read() throws IOException {
        String line;
        while ((line = reader.readLine()) != null) {
            // Skip lines starting with a #
            if (line.charAt(0) != '#') {
                try {
                    ConstraintSyntaxTree cst = parseLine(line, 0);
                    Constraint constraint = new Constraint(model);
                    constraint.setConsSyntax(cst);
                    model.add(constraint);
                } catch (Exception e) {
                    
                }
            }
        }
        
        return model;
    }
}
