package de.uni_hildesheim.sse.trans;

import java.io.File;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import de.uni_hildesheim.sse.model.varModel.IvmlKeyWords;
import de.uni_hildesheim.sse.trans.in.InputType;
import de.uni_hildesheim.sse.trans.out.OutputType;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.LoggingLevel;

/**
 * Starting point for the whole translation.
 * @author El-Sharkawy
 *
 */
@SuppressWarnings("static-access")
public class Main {
    
    private static final Options OPTIONS = new Options();
    
    private static final String CMD_IN = "in";
    private static final String CMD_OUT = "out";
    private static final String CMD_COMMENT = "comment";
    
    static {
        createOption(CMD_IN, "The input file which shall be translated", CMD_IN);
        createOption(CMD_OUT, "The output file, where the translated model shall be saved", CMD_OUT);
        createOption(CMD_COMMENT, "An optional comment, which shall be added to the translated result", CMD_COMMENT);
        EASyLoggerFactory.INSTANCE.setLoggingLevel(LoggingLevel.INFO);
    }
    
    /**
     * Creates and option with an argument.
     * @param argName The command which shall be passed to the passed to this application, e.g -in
     * @param description A description for the user.
     * @param cmd The unique identifier of the option. 
     */
    private static void createOption(String argName, String description, String cmd) {
        Option option = OptionBuilder.withArgName(argName).hasArg().withDescription(description).create(cmd);
        OPTIONS.addOption(option);
    }

    /**
     * Main-Method of this translator.
     * @param args Must not be <tt>null</tt> or empty. Must be in the following from:
     * <ul>
     * <li><tt>args.length >= 4:</tt></li>
     *   <ul>
     *   <li>-i &lt;input file&gt; -o &lt;output file&gt;: This option tries to detect the correct translation settings
     *   automatically, depending on the file extensions</li>
     *   </ul>
     * </ul>
     */
    public static void main(String[] args) {
        CommandLineParser parser = new BasicParser();
        
        if (args == null || args.length < 2) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("ModelTranslator", OPTIONS);
        } else {
            File inputFile = null;
            File outputFile = null;
            String comment = null;
            try {
                CommandLine cmd = parser.parse(OPTIONS, args);
                if (cmd.hasOption(CMD_IN)) {
                    inputFile = new File(cmd.getOptionValue(CMD_IN));
                }
                if (cmd.hasOption(CMD_OUT)) {
                    outputFile = new File(cmd.getOptionValue(CMD_OUT));
                }
                if (cmd.hasOption(CMD_COMMENT)) {
                    comment = "\\\\" + cmd.getOptionValue(CMD_COMMENT) + IvmlKeyWords.LINEFEED;
                }
                ModelTranslator.translate(inputFile, outputFile, InputType.MODEL, OutputType.IVML, comment);
            } catch (ParseException exp) {
                System.err.println("Parsing failed. Reason: " + exp.getMessage());
    
            }
        }
    }
}
