package de.uni_hildesheim.sse.trans;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;

import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.trans.in.IModelReader;
import de.uni_hildesheim.sse.trans.in.InputType;
import de.uni_hildesheim.sse.trans.in.ModelReader;
import de.uni_hildesheim.sse.trans.out.IModelWriter;
import de.uni_hildesheim.sse.trans.out.IVMLWriter;
import de.uni_hildesheim.sse.trans.out.OutputType;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * ModelTranslator utility class, responsible for translating models into another file format.
 * @author El-Sharkawy
 *
 */
public class ModelTranslator {
    
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(ModelTranslator.class, Bundle.ID);
    
    /**
     * Should avoid instantiation of this utility class.
     */
    private ModelTranslator() {}
    
    /**
     * Translates a model into another file format.
     * @param in The input model, which shall be translated.
     * @param out The destination of the translated model.
     * @param inType The type of <tt>in</tt>. Must be one of {@link InputType}.
     * @param outType Specification in which format <tt>in</tt> shall be translated. Must be one of {@link OutputType}.
     * @param comment An optional comment (e.g. information about the translated model), can be <tt>null</tt>.
     */
    public static void translate(File in, Writer out, InputType inType, OutputType outType, String comment) {
        Project model = null;

        LOGGER.info("1.) Read model."); 
        IModelReader reader = null;
        switch (inType) {
        case MODEL:
            try {
                reader = new ModelReader(in);
                model = reader.read();
            } catch (IOException e) {
                LOGGER.exception(e);
            }
            break;
        default:
            LOGGER.debug("Not supported input model specified.");
            break;
        }
        
        if (null != model) {
            LOGGER.info("2.) Write model."); 
            IModelWriter writer = null;
            switch (outType) {
            case IVML:
                writer = new IVMLWriter(out, model);
                writer.write(comment);
                try {
                    out.flush();
                    out.close();
                } catch (IOException e) {
                    LOGGER.exception(e);
                }
                break;
            default:
                LOGGER.debug("Not supported input model specified.");
                break;
            }
        }
    }
    
    /**
     * Translates a model into another file format.
     * Default method which should be used.
     * @param in The input model, which shall be translated.
     * @param out The destination of the translated model.
     * @param inType The type of <tt>in</tt>. Must be one of {@link InputType}.
     * @param outType Specification in which format <tt>in</tt> shall be translated. Must be one of {@link OutputType}.
     * @param comment An optional comment (e.g. information about the translated model), can be <tt>null</tt>.
     */
    public static void translate(File in, File out, InputType inType, OutputType outType, String comment) {
        try {
            FileWriter outWriter = new FileWriter(out);
            translate(in, outWriter, inType, outType, comment);
        } catch (IOException e) {
            LOGGER.exception(e);
        }
    }

}
