package de.uni_hildesheim.sse.trans;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;

import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.trans.convert.CNFConverter;
import de.uni_hildesheim.sse.trans.convert.MaxTermConverter;
import de.uni_hildesheim.sse.trans.in.IModelReader;
import de.uni_hildesheim.sse.trans.in.InputType;
import de.uni_hildesheim.sse.trans.in.ModelReader;
import de.uni_hildesheim.sse.trans.out.DimacsWriter;
import de.uni_hildesheim.sse.trans.out.IModelWriter;
import de.uni_hildesheim.sse.trans.out.IVMLWriter;
import de.uni_hildesheim.sse.trans.out.OutputType;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * ModelTranslator utility class, responsible for translating models into another file format.
 * @author El-Sharkawy
 *
 */
public class ModelTranslator {
    
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(ModelTranslator.class, Bundle.ID);
    
    /**
     * Should avoid instantiation of this utility class.
     */
    private ModelTranslator() {}
    
    // checkstyle: stop parameter number check
    
    /**
     * Translates a model into another file format.
     * @param in The input model, which shall be translated.
     * @param out The destination of the translated model.
     * @param inType The type of <tt>in</tt>. Must be one of {@link InputType}.
     * @param outType Specification in which format <tt>in</tt> shall be translated. Must be one of {@link OutputType}.
     * @param comment An optional comment (e.g. information about the translated model), can be <tt>null</tt>.
     * @param version Optionally the version of the translated model. Can be <tt>null</tt>. If not <tt>null</tt>,
     *     it must be in {@link de.uni_hildesheim.sse.utils.modelManagement.Version#Version(String)} syntax.
     */
    public static void translate(File in, Writer out, InputType inType, OutputType outType, String comment,
        String version) {
        
        Project model = null;

        LOGGER.info("1.) Read model."); 
        IModelReader reader = null;
        switch (inType) {
        case MODEL:
            try {
                reader = new ModelReader(in);
                model = reader.read();
            } catch (IOException e) {
                LOGGER.exception(e);
            }
            break;
        default:
            LOGGER.debug("Not supported input model specified.");
            break;
        }
        
        if (null != model) {
            LOGGER.info("2.) Write model."); 
            IModelWriter writer = null;
            switch (outType) {
            case IVML:
                writer = new IVMLWriter(out, model);
                writer.write(comment, version);
                try {
                    out.flush();
                    out.close();
                } catch (IOException e) {
                    LOGGER.exception(e);
                }
                break;
            case DIMCAS:
                CNFConverter.convert(model, new MaxTermConverter());
                System.out.println("Conversion completed.");
                writer = new DimacsWriter(model, out);
                writer.write(comment, version);
                try {
                    out.flush();
                    out.close();
                } catch (IOException e) {
                    LOGGER.exception(e);
                }
                break;
            default:
                LOGGER.debug("Not supported input model specified.");
                break;
            }
        }
    }
    
    /**
     * Translates a model into another file format.
     * Default method which should be used.
     * @param in The input model, which shall be translated.
     * @param out The destination of the translated model.
     * @param inType The type of <tt>in</tt>. Must be one of {@link InputType}.
     * @param outType Specification in which format <tt>in</tt> shall be translated. Must be one of {@link OutputType}.
     * @param comment An optional comment (e.g. information about the translated model), can be <tt>null</tt>.
     * @param version Optionally the version of the translated model. Can be <tt>null</tt>. If not <tt>null</tt>,
     *     it must be in {@link de.uni_hildesheim.sse.utils.modelManagement.Version#Version(String)} syntax.
     */
    public static void translate(File in, File out, InputType inType, OutputType outType, String comment,
        String version) {
        
        try {
            FileWriter outWriter = new FileWriter(out);
            translate(in, outWriter, inType, outType, comment, version);
        } catch (IOException e) {
            LOGGER.exception(e);
        }
    }

    // checkstyle: resume parameter number check
}
