package de.uni_hildesheim.sse.trans.out;

import java.io.IOException;
import java.io.Writer;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.IvmlKeyWords;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.filter.ConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder.VisibilityType;
import de.uni_hildesheim.sse.model.varModel.filter.FilterType;
import de.uni_hildesheim.sse.trans.Bundle;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Writer to write the given project into the DIMACS format.
 * Only works on projects that have their constraints in CNF.
 *
 * @author Adam Krafcyk
 */
public class DimacsWriter implements IModelWriter {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(DimacsWriter.class, Bundle.ID);
    
    private Project project;
    private Map<String, Integer> variables;
    private Writer writer;
    
    /**
     * Creates the writer for the given project.
     * @param project The project which shall be saved.
     * @param writer Writer which should be used for writing the output.
     */
    public DimacsWriter(Project project, Writer writer) {
        this.project = project;
        this.writer = writer;
        variables = new HashMap<String, Integer>();
    }
    
    /**
     * Writes the DIMACS format to the {@link Writer}.
     * @throws IOException If writing to the stream fails
     */
    public void write() throws IOException {
        // Get all variables
        DeclarationFinder declarationFinder = new DeclarationFinder(project, FilterType.ALL, null);
        List<AbstractVariable> variableDeclarations = declarationFinder.getVariableDeclarations(VisibilityType.ALL);
        Collections.sort(variableDeclarations, new Comparator<AbstractVariable>() {
            public int compare(AbstractVariable o1, AbstractVariable o2) {
                return o1.getName().compareTo(o2.getName());
            }
        });
        
        // Get all Constraints
        ConstraintFinder constraintFinder = new ConstraintFinder(project);
        List<Constraint> constraints = constraintFinder.getConstraints();
        
        // Write variables
        Integer i = 1;
        for (AbstractVariable variable : variableDeclarations) {
            variables.put(variable.getName(), i++);
            writer.append("c ");
            writer.append(i.toString());
            writer.append(" ");
            writer.append(variable.getName());
            writer.append(IvmlKeyWords.LINEFEED);
        }
        
        writer.append("p CNF " + variableDeclarations.size() + " " + constraints.size());
        writer.append(IvmlKeyWords.LINEFEED);
        
        for (Constraint constraint : constraints) {
            ConstraintSyntaxTree tree = constraint.getConsSyntax();
            writeTree(tree);
        }
        
        writer.flush();
    }
    
    /**
     * Writes the {@link ConstraintSyntaxTree} in the DIMACS format.
     * Assumes that the {@link ConstraintSyntaxTree} is in CNF.
     * @param tree The {@link ConstraintSyntaxTree} to be written.
     * @throws IOException When writing to the {@link Writer} fails
     */
    private void writeTree(ConstraintSyntaxTree tree) throws IOException {
        
    }

    @Override
    public void write(String comment, String version) {
        try {
            if (comment != null) {
                writer.append("c ");
                writer.append(comment);
                writer.append(IvmlKeyWords.LINEFEED);
            }
            if (version != null) {
                writer.append("c Version ");
                writer.append(version);
                writer.append(IvmlKeyWords.LINEFEED);
            }
            write();
        } catch (IOException e) {
            LOGGER.exception(e);
        }
    }
    
        /*DeclarationFinder declarationFinder = new DeclarationFinder(project, FilterType.ALL, null);
        List<AbstractVariable> variableDeclarations = declarationFinder.getVariableDeclarations(VisibilityType.ALL);
        Collections.sort(variableDeclarations, new Comparator<AbstractVariable>() {
            public int compare(AbstractVariable o1, AbstractVariable o2) {
                return o1.getName().compareTo(o2.getName());
            }
        });
        
        Integer i = 1;
        for (AbstractVariable variable : variableDeclarations) {
            variables.put(variable.getName(), i++);
            out.print("c ");
            out.print(i.toString());
            out.print(" ");
            out.print(variable.getName());
            out.println();
        }
        
        // TODO print(p CNF numVars numConstrains);
        
        ConstraintFinder constraintFinder = new ConstraintFinder(project);
        
        Configuration config = new Configuration(project);
        EvaluationVisitor evalVisitor = new EvaluationVisitor();
        // Assignment state = null avoids value assignment in configuration
        evalVisitor.init(config, null, false, null);
        
        int anz = 0;
        for (Constraint constraint : constraintFinder.getConstraints()) {
            anz++;
            if (anz > 100) {
                break;
            }
            
            DeclrationInConstraintFinder varFinder = new DeclrationInConstraintFinder(constraint.getConsSyntax());
            
            Object[] vars = varFinder.getDeclarations().toArray();
            
            int bitmask = 0;
            for (int k = 0; k < Math.pow(2, vars.length); k++) {
                bitmask++;
                boolean[] values = new boolean[vars.length];
                for (int j = 0; j < vars.length; j++) {
                    IDecisionVariable decisionVar = config.getDecision((AbstractVariable) vars[j]);
                    try {
                        boolean bool = (bitmask & 1 << j) != 0;
                        values[j] = bool;
                        decisionVar.setValue(bool ? BooleanValue.TRUE : BooleanValue.FALSE, AssignmentState.ASSIGNED);
                    } catch (ConfigurationException e) {
                        e.printStackTrace();
                    }
                }
                
                boolean result = Math.random() >= 0.5; // TODO
                
                if (result) {
                    for (int j = 0; j < vars.length; j++) {
                        if (!values[j]) {
                            out.print("-");
                        }
                        out.print(variables.get(((AbstractVariable) vars[j]).getName()));
                        out.print(" ");
                    }
                    out.println("0");
                }
            }
            
            
            //out.print(constraint.toString());
            //out.println();
        }
        out.flush();*/
    
}
