package de.uni_hildesheim.sse.trans.convert;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import de.uni_hildesheim.sse.model.confModel.AssignmentState;
import de.uni_hildesheim.sse.model.confModel.Configuration;
import de.uni_hildesheim.sse.model.confModel.ConfigurationException;
import de.uni_hildesheim.sse.model.confModel.IDecisionVariable;
import de.uni_hildesheim.sse.model.cst.CSTSemanticException;
import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.cstEvaluation.EvaluationVisitor;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.model.varModel.filter.DeclrationInConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.values.BooleanValue;
import de.uni_hildesheim.sse.trans.Bundle;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Converts a boolean formula in a list of disjunction terms by means of
 * <a href="http://de.wikipedia.org/wiki/Konjunktive_Normalform#Beispiel_f.C3.BCr_die_Bildung">max terms</a>.
 * @author Adam Krafczyk
 * @author El-Sharkawy
 *
 */
public class MaxTermConverter implements ICNFConvertStrategy {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(MaxTermConverter.class, Bundle.ID);

    private Project project;
    
    @Override
    public void convert(Constraint constraint) {
        project = (Project) constraint.getTopLevelParent();
        
        List<ConstraintSyntaxTree> parts = createCNFParts(constraint);
        for (ConstraintSyntaxTree part : parts) {
            addConstraint(part);
        }
    }
    
    /**
     * Creates a list of parts of CNF for the given {@link Constraint}.
     * @param constraint A boolean expression
     * @return List of parts of CNF (boolean expressions with OR and NOT).
     */
    private List<ConstraintSyntaxTree> createCNFParts(Constraint constraint) {
        // Get an array of all variables in the constraint
        ConstraintSyntaxTree originalConstraint = constraint.getConsSyntax();
        DeclrationInConstraintFinder finder = new DeclrationInConstraintFinder(originalConstraint);
        Set<AbstractVariable> declarations = finder.getDeclarations();
        AbstractVariable[] declarationArray = declarations.toArray(new AbstractVariable[] {});
        Arrays.sort(declarationArray, new Comparator<AbstractVariable>() {
            public int compare(AbstractVariable o1, AbstractVariable o2) {
                return o1.getName().compareTo(o2.getName());
            }
        });
        
        // Create a project which only contains our single Constraint
        Project singleConstraintProject = new Project("SingleConstraintProject");
        Constraint constraintCopy = null;
        try {
            constraintCopy = new Constraint(originalConstraint, singleConstraintProject);
        } catch (CSTSemanticException e) {
            // TODO (can this even happen?)
            LOGGER.exception(e);
        }
        singleConstraintProject.add(constraintCopy);
        for (AbstractVariable var : declarationArray) {
            singleConstraintProject.add(var);
        }
        
        // Create a configuration object for the singleConstraintProject
        Configuration config = new Configuration(singleConstraintProject);
        
        List<ConstraintSyntaxTree> parts = new ArrayList<ConstraintSyntaxTree>();
        // for each combination of the variables (true or false)
        // TODO: only works for up to 16 vars
        if (declarationArray.length >= 16) {
            System.out.println("Skipping too long Constraint");
        } else {
            handleSmallConstraint(originalConstraint, declarationArray, config, parts);
        }
        
        return parts;
    }

    /**
     * Converts a small constraint into a CNF formula.
     * This constraint must have at maximum 15 different {@link AbstractVariable}s.
     * @param originalConstraint The constraint which shall be converted into CNF formula.
     *     this constraint may consists only of AND, OR, and NOT operations.
     * @param declarationArray The {@link AbstractVariable}s, which are used inside the <tt>originalConstraint</tt>.
     * @param config A temporary {@link Configuration} containing only <tt>originalConstraint</tt>
     *     and <tt>declarationArray</tt>
     * @param parts An empty list, where the resulting disjunctions terms of the CNF formula shall be added to
     */
    protected void handleSmallConstraint(ConstraintSyntaxTree originalConstraint, AbstractVariable[] declarationArray,
        Configuration config, List<ConstraintSyntaxTree> parts) {
        
        EvaluationVisitor evalVisitor = new EvaluationVisitor();
        evalVisitor.init(config, null, false, null);
        for (int i = 0; i < Math.pow(2, declarationArray.length); i++) {
            boolean[] state = new boolean[declarationArray.length];
            for (int j = 0; j < state.length; j++) {
                IDecisionVariable decisionVar = config.getDecision(declarationArray[j]);
                if ((i & (1 << j)) != 0) {
                    state[j] = true;
                    try {
                        decisionVar.setValue(BooleanValue.TRUE, AssignmentState.ASSIGNED);
                    } catch (ConfigurationException e) {
                        // TODO
                        LOGGER.exception(e);
                    }
                } else {
                    state[j] = false;
                    try {
                        decisionVar.setValue(BooleanValue.FALSE, AssignmentState.ASSIGNED);
                    } catch (ConfigurationException e) {
                        // TODO
                        LOGGER.exception(e);
                    }
                }
            }
            
            // get the result
            originalConstraint.accept(evalVisitor);
            boolean result = evalVisitor.constraintFulfilled();
            
            // if the result it false, add the negated combination to the list of expressions
            if (!result) {
                parts.add(createNegatedORExpression(declarationArray, state)); 
            }
            
            // 1 * init() + n * clearResult() is much faster than n * (init() + clear())
            evalVisitor.clearResult();
        }
    }
    
//    /**
//     * Returns whether the boolean expression with the given variable states is true or false.
//     * @param expression The boolean expression
//     * @param config The {@link Configuration} that defines the state of the variables
//     * @return The result of the boolean expression
//     */
//    private boolean getResult(ConstraintSyntaxTree expression, Configuration config) {
//        EvaluationVisitor evalVisitor = new EvaluationVisitor();
//        evalVisitor.init(config, null, false, null);
//        expression.accept(evalVisitor);
//        BooleanValue result = (BooleanValue) evalVisitor.getResult();
//        return result.equals(BooleanValue.TRUE);
//    }
    
    /**
     * Creates an {@link ConstraintSyntaxTree} with the variables negated and OR'd together.
     * @param variables Array of Variables
     * @param states Array whether each variable is true or false
     * @return the {@link ConstraintSyntaxTree}
     */
    private ConstraintSyntaxTree createNegatedORExpression(AbstractVariable[] variables, boolean[] states) {
        ConstraintSyntaxTree call = null;
        if (!states[0]) {
            call = new Variable(variables[0]);
        } else {
            call = new OCLFeatureCall(new Variable(variables[0]), OclKeyWords.NOT);
        }
        
        for (int i = 1; i < variables.length; i++) {
            ConstraintSyntaxTree variable = new Variable(variables[i]);
            if (states[i]) {
                variable = new OCLFeatureCall(variable, OclKeyWords.NOT);
            }
            call = new OCLFeatureCall(call, OclKeyWords.OR, variable);
        }
        return call;
    }

    /**
     * Adds the given constraint to the project.
     * @param cst The constraint to add.
     */
    private void addConstraint(ConstraintSyntaxTree cst) {
        Constraint constraint = new Constraint(project);
        try {
            constraint.setConsSyntax(cst);
        } catch (CSTSemanticException e) {
            LOGGER.exception(e);
        }
        
        project.add(constraint);
    }
}
