package de.uni_hildesheim.sse.trans.convert;

import java.util.List;

import de.uni_hildesheim.sse.model.confModel.AssignmentState;
import de.uni_hildesheim.sse.model.confModel.Configuration;
import de.uni_hildesheim.sse.model.confModel.ConfigurationException;
import de.uni_hildesheim.sse.model.confModel.IDecisionVariable;
import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.cstEvaluation.EvaluationVisitor;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.model.varModel.values.BooleanValue;
import de.uni_hildesheim.sse.trans.Bundle;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Alternative implementation of {@link MaxTermConverter}, which uses a backtracking algorithm
 * instead of iteration to find all max terms.
 * 
 * @author Adam Krafczyk
 */
public class MaxTermConverter2 extends MaxTermConverter {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(MaxTermConverter2.class, Bundle.ID);


    @Override
    protected void handleSmallConstraint(ConstraintSyntaxTree originalConstraint, AbstractVariable[] declarationArray,
        Configuration config, List<ConstraintSyntaxTree> parts) {
        
        handleConstraint(originalConstraint, declarationArray, config, parts, 0);
    }
    
    @Override
    protected void handleBigConstraint(ConstraintSyntaxTree originalConstraint, AbstractVariable[] declarationArray,
        Configuration config, List<ConstraintSyntaxTree> parts) {
        
        handleConstraint(originalConstraint, declarationArray, config, parts, 0);
    }
    
    /**
     * Converts a constraint into a CNF formula.
     * @param originalConstraint The constraint which shall be converted into CNF formula.
     *     this constraint may consists only of AND, OR, and NOT operations.
     * @param declarationArray The {@link AbstractVariable}s, which are used inside the <tt>originalConstraint</tt>.
     * @param config A temporary {@link Configuration} containing only <tt>originalConstraint</tt>
     *     and <tt>declarationArray</tt>
     * @param parts An empty list, where the resulting disjunctions terms of the CNF formula shall be added to
     * @param var the variable to change
     */
    protected void handleConstraint(ConstraintSyntaxTree originalConstraint,
            AbstractVariable[] declarationArray, Configuration config, List<ConstraintSyntaxTree> parts, int var) {
        
        if (var >= declarationArray.length) {
            return;
        }
        
        try {
            IDecisionVariable decisionVar = config.getDecision(declarationArray[var]);
            
            decisionVar.setValue(BooleanValue.TRUE, AssignmentState.ASSIGNED);
            if (isFalse(originalConstraint, config)) {
                addExpression(parts, declarationArray, var, config);
            } else {
                handleConstraint(originalConstraint, declarationArray, config, parts, var + 1);
            }
            
            decisionVar.setValue(BooleanValue.FALSE, AssignmentState.ASSIGNED);
            if (isFalse(originalConstraint, config)) {
                addExpression(parts, declarationArray, var, config);
            } else {
                handleConstraint(originalConstraint, declarationArray, config, parts, var + 1);
            }
            
            decisionVar.setValue(null, AssignmentState.UNDEFINED);
            
        } catch (ConfigurationException e) {
            LOGGER.exception(e);
        }
    }
    
    /**
     * Tests if the boolean expression is false.
     * 
     * @param constraint The {@link ConstraintSyntaxTree} that represents the boolean expression
     * @param config the {@link Configuration} that holds the variable states
     * 
     * @return whether the boolean expression is false
     */
    private boolean isFalse(ConstraintSyntaxTree constraint, Configuration config) {
        EvaluationVisitor evaluationVisitor = new EvaluationVisitor(config, null, false, null);
        
        constraint.accept(evaluationVisitor);
        boolean isFalse = !evaluationVisitor.constraintUndefined() && evaluationVisitor.constraintFailed();
        //evaluationVisitor.clear();
        
        return isFalse;
    }
    
    /**
     * Adds the boolean expression to the list of expressions.
     * 
     * @param parts The List that the new part is added to
     * @param declarationArray Array with all variable declarations
     * @param var The number of variables that is defined (0 to var -1 in the declaration array)
     * @param config The {@link Configuration} that holds the states of the variables
     */
    private void addExpression(List<ConstraintSyntaxTree> parts, AbstractVariable[] declarationArray, int var,
            Configuration config) {
        
        boolean[] state = new boolean[var + 1];
        for (int i = 0; i < state.length; i++) {
            state[i] = ((BooleanValue) config.getDecision(declarationArray[i]).getValue()) == BooleanValue.TRUE;
        }
        parts.add(createNegatedORExpression(declarationArray, state));
    }
    
    /**
     * Creates an {@link ConstraintSyntaxTree} with the variables negated and OR'd together.
     * @param variables Array of Variables
     * @param states Array whether each variable is true or false
     * @return the {@link ConstraintSyntaxTree}
     */
    private ConstraintSyntaxTree createNegatedORExpression(AbstractVariable[] variables, boolean[] states) {
        ConstraintSyntaxTree call = null;
        if (!states[0]) {
            call = new Variable(variables[0]);
        } else {
            call = new OCLFeatureCall(new Variable(variables[0]), OclKeyWords.NOT);
        }
        
        for (int i = 1; i < states.length; i++) {
            ConstraintSyntaxTree variable = new Variable(variables[i]);
            if (states[i]) {
                variable = new OCLFeatureCall(variable, OclKeyWords.NOT);
            }
            call = new OCLFeatureCall(call, OclKeyWords.OR, variable);
        }
        return call;
    }
    
}
