package de.uni_hildesheim.sse.trans.in.rsf;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.trans.in.AbstractReader;
import de.uni_hildesheim.sse.trans.in.ParserException;
import de.uni_hildesheim.sse.trans.in.ParserException.ParserExceptionType;

/**
 * Represents a condition read from a .rsf file.
 * 
 * @author Adam Krafczyk
 */
abstract class RSFCondition {

    /**
     * Returns this condition as a {@link ConstraintSyntaxTree}.
     * @param reader The reader to get the variables from.
     * @return A {@link ConstraintSyntaxTree} representing this condition.
     * @throws ParserException If the conditions contains an operation, that is not supported by the parser.
     */
    abstract ConstraintSyntaxTree toConstraintSyntaxTree(AbstractReader reader) throws ParserException;
    
    /**
     * Converts the given variable from a string to a {@link ConstraintSyntaxTree}.
     * A ! at the beginning will result in a {@link OCLFeatureCall} with NOT.
     * @param reader The reader to read variable declarations from.
     * @param var The string representing a variable.
     * @return The variable as {@link ConstraintSyntaxTree}.
     */
    private ConstraintSyntaxTree getVariable(AbstractReader reader, String var) {
        ConstraintSyntaxTree result = null;
        if (var.startsWith("!")) {
            Variable variable = new Variable(reader.getVariable(var.substring(1)));
            result = new OCLFeatureCall(variable, OclKeyWords.NOT);
        } else {
            result = new Variable(reader.getVariable(var));
        }
        return result;
    }
    
    /**
     * Converts the condition into a {@link ConstraintSyntaxTree}.
     * @param reader The reader to get the variables from.
     * @param condition A string representing the condition as read from a .rsf file.
     * @return A {@link ConstraintSyntaxTree} representing the condition.
     * @throws ParserException If the conditions contains an operation, that is not supported by the parser.
     * 
     * TODO handle constants.
     */
    protected ConstraintSyntaxTree getConstraintSyntaxTree(AbstractReader reader, String condition)
        throws ParserException {
        
        condition = condition.replace(" ", "");
        
        // We assume that ! can only be in front of variables
        
        String highestOperand = null;
        int highestOperandLevel = Integer.MAX_VALUE;
        int highestOperandPos = -1;
        int currentLevel = 0;
        
        for (int i = 0; i < condition.length(); i++) {
            char c = condition.charAt(i);
            switch (c) {
            case '(':
                currentLevel++;
                break;
            case ')':
                currentLevel--;
                break;
                
            case '&':
                if (condition.charAt(i + 1) != '&') {
                    throw new ParserException(ParserExceptionType.NOT_SUPPORTED_FIELD); // TODO
                }
                if (highestOperandLevel > currentLevel) {
                    highestOperand = OclKeyWords.AND;
                    highestOperandPos = i;
                    highestOperandLevel = currentLevel;
                }
                i++;
                break;
            
            case '|':
                if (condition.charAt(i + 1) != '|') {
                    throw new ParserException(ParserExceptionType.NOT_SUPPORTED_FIELD); // TODO
                }
                if (highestOperandLevel > currentLevel) {
                    highestOperand = OclKeyWords.OR;
                    highestOperandPos = i;
                    highestOperandLevel = currentLevel;
                }
                i++;
                break;
            
            default:
                break;
            }
        }
        
        ConstraintSyntaxTree result = null;
        
        if (highestOperandPos == -1) {
            // we only have a variable
            result = getVariable(reader, condition);
        } else {
            StringBuffer left = new StringBuffer();
            StringBuffer right = new StringBuffer();
            split(condition, highestOperandPos, highestOperandLevel, left, right);
            result = new OCLFeatureCall(getConstraintSyntaxTree(reader, left.toString()), highestOperand,
                    getConstraintSyntaxTree(reader, right.toString()));
        }
        
        return result;
    }
    
    /**
     * Splits the given line at the given operand position into a left and right part.
     * @param line The whole line
     * @param operandPos The position of the operand in the line
     * @param operandLevel The logic level of the operand (by counting brackets)
     * @param left A {@link StringBuffer} where the left side of the operand will be written to
     * @param right A {@link StringBuffer} where the right side of the operand will be written to
     */
    private void split(String line, int operandPos, int operandLevel, StringBuffer left, StringBuffer right) {
        left.append(line.substring(0, operandPos));
        right.append(line.substring(operandPos + 2, line.length()));
        int currentLevel = 0;
        for (int i = 0; i < left.length(); i++) {
            char c = left.charAt(i);
            if ((c == '(' || c == ')') && currentLevel < operandLevel) {
                left.replace(i, i + 1, "");
            }
            if (c == '(') {
                currentLevel++;
            }
            if (c == ')') {
                currentLevel--;
            }
        }
        currentLevel = 0;
        for (int i = right.length() - 1; i >= 0; i--) {
            char c = right.charAt(i);
            if ((c == '(' || c == ')') && currentLevel < operandLevel) {
                right.replace(i, i + 1, "");
            }
            if (c == ')') {
                currentLevel++;
            }
            if (c == '(') {
                currentLevel--;
            }
        }
    }
    
}
