package de.uni_hildesheim.sse.trans.convert;

import java.util.List;

import de.uni_hildesheim.sse.model.cst.ConstraintSyntaxTree;
import de.uni_hildesheim.sse.model.cst.OCLFeatureCall;
import de.uni_hildesheim.sse.model.cst.Variable;
import de.uni_hildesheim.sse.model.varModel.AbstractVariable;
import de.uni_hildesheim.sse.model.varModel.Constraint;
import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.model.varModel.datatypes.OclKeyWords;
import de.uni_hildesheim.sse.model.varModel.filter.ConstraintFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder;
import de.uni_hildesheim.sse.model.varModel.filter.DeclarationFinder.VisibilityType;
import de.uni_hildesheim.sse.model.varModel.filter.FilterType;
import de.uni_hildesheim.sse.trans.Bundle;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * Checks a given model for unused variables (i.e. variables that are not used in constraints).
 * 
 * @author Adam Krafczyk
 */
public class UnusedVariableChecker {
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(
            UnusedVariableChecker.class, Bundle.ID);

    private Project model;
    
    /**
     * Creates a {@link UnusedVariableChecker} for the given project.
     * @param model The project to be checked by this checker
     */
    public UnusedVariableChecker(Project model) {
        this.model = model;
    }
    
    /**
     * Removes all unused variables (i.e. variables that are not used in constraints) from the project.
     * 
     * @return The number of removed variables.
     */
    public int removeUnusedVariables() {
        ConstraintFinder cFinder = new ConstraintFinder(model);
        DeclarationFinder varFinder = new DeclarationFinder(model, FilterType.ALL, null);
        
        List<AbstractVariable> varList = varFinder.getVariableDeclarations(VisibilityType.ALL);
        
        for (Constraint constraint : cFinder.getConstraints()) {
            removeVariables(constraint.getConsSyntax(), varList);
        }
        
        for (AbstractVariable var : varList) {
            model.removeElement(var);
        }
        
        return varList.size();
    }
    
    /**
     * Searches all variables in the {@link ConstraintSyntaxTree} and removes them from the variable list.
     * Assumes the {@link ConstraintSyntaxTree} to only consist of variables, ANDs, ORs and NOTs.
     * @param tree The {@link ConstraintSyntaxTree} where variables are searched
     * @param variables The list where variables are removed
     */
    private void removeVariables(ConstraintSyntaxTree tree, List<AbstractVariable> variables) {
        if (tree instanceof Variable) {
            AbstractVariable variable = ((Variable) tree).getVariable();
            variables.remove(variable);
        } else if (tree instanceof OCLFeatureCall) {
            OCLFeatureCall call = (OCLFeatureCall) tree;
            if (call.getOperation().equals(OclKeyWords.OR) || call.getOperation().equals(OclKeyWords.AND)) {
                removeVariables(call.getParameter(0), variables);
                removeVariables(call.getOperand(), variables);
            } else if (call.getOperation().equals(OclKeyWords.NOT)) {
                removeVariables(call.getOperand(), variables);
            } else {
                LOGGER.debug("Unexpected Operation in model... skipping"); // TODO
            }
        } else {
            LOGGER.debug("Unexpected ConstraintSyntaxTree in model... skipping"); // TODO
        }
    }
    
}
