package de.uni_hildesheim.sse.trans;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;

import de.uni_hildesheim.sse.model.varModel.Project;
import de.uni_hildesheim.sse.trans.convert.CNFConverter;
import de.uni_hildesheim.sse.trans.convert.MaxTermConverter2;
import de.uni_hildesheim.sse.trans.convert.ModelOptimizer;
import de.uni_hildesheim.sse.trans.in.IModelReader;
import de.uni_hildesheim.sse.trans.in.InputType;
import de.uni_hildesheim.sse.trans.in.ModelReader;
import de.uni_hildesheim.sse.trans.in.ParserException;
import de.uni_hildesheim.sse.trans.in.rsf.RSFReader;
import de.uni_hildesheim.sse.trans.out.DimacsWriter;
import de.uni_hildesheim.sse.trans.out.IModelWriter;
import de.uni_hildesheim.sse.trans.out.IVMLWriter;
import de.uni_hildesheim.sse.trans.out.OutputType;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory;
import de.uni_hildesheim.sse.utils.logger.EASyLoggerFactory.EASyLogger;

/**
 * ModelTranslator utility class, responsible for translating models into another file format.
 * @author El-Sharkawy
 *
 */
public class ModelTranslator {
    
    private static final EASyLogger LOGGER = EASyLoggerFactory.INSTANCE.getLogger(ModelTranslator.class, Bundle.ID);
    
    /**
     * Should avoid instantiation of this utility class.
     */
    private ModelTranslator() {}
    
    // checkstyle: stop parameter number check
    
    /**
     * Translates a model into another file format.
     * @param in The input model, which shall be translated.
     * @param out The destination of the translated model.
     * @param inType The type of <tt>in</tt>. Must be one of {@link InputType}.
     * @param outType Specification in which format <tt>in</tt> shall be translated. Must be one of {@link OutputType}.
     * @param comment An optional comment (e.g. information about the translated model), can be <tt>null</tt>.
     * @param version Optionally the version of the translated model. Can be <tt>null</tt>. If not <tt>null</tt>,
     *     it must be in {@link de.uni_hildesheim.sse.utils.modelManagement.Version#Version(String)} syntax.
     */
    public static void translate(File in, Writer out, InputType inType, OutputType outType, String comment,
        String version) {
        
        Project model = readSourceModel(in, inType);
        transformAndWriteModel(out, outType, comment, version, model);
    }

    /**
     * Transforms and writes the model to the specified format.
     * @param out The destination of the translated model.
     * @param outType Specification in which format <tt>in</tt> shall be translated. Must be one of {@link OutputType}.
     * @param comment An optional comment (e.g. information about the translated model), can be <tt>null</tt>.
     * @param version Optionally the version of the translated model. Can be <tt>null</tt>. If not <tt>null</tt>,
     *     it must be in {@link de.uni_hildesheim.sse.utils.modelManagement.Version#Version(String)} syntax.
     * @param model The read model.
     */
    private static void transformAndWriteModel(Writer out, OutputType outType, String comment, String version,
            Project model) {
        if (null != model) {
            IModelWriter writer = null;
            switch (outType) {
            case IVML:
                LOGGER.info("2.) Write model."); 
                writer = new IVMLWriter(out, model);
                writer.write(comment, version);
                try {
                    out.flush();
                    out.close();
                } catch (IOException e) {
                    LOGGER.exception(e);
                }
                break;
            case DIMCAS:
                LOGGER.info("2.) Convert model."); 
                CNFConverter.convert(model, new MaxTermConverter2());
                System.out.println("Conversion completed.");
                
                LOGGER.info("3.) Optimize model."); 
                ModelOptimizer optimizer = new ModelOptimizer(model);
                int removed = optimizer.removeUnusedVariables();
                System.out.println("Removed " + removed + " variables.");
                removed = optimizer.handleConstantVariables();
                System.out.println("Removed " + removed + " constraints.");
                System.out.println("Optimizing completed.");
                
                LOGGER.info("4.) Write model."); 
                writer = new DimacsWriter(model, out);
                writer.write(comment, version);
                try {
                    out.flush();
                    out.close();
                } catch (IOException e) {
                    LOGGER.exception(e);
                }
                System.out.println("Writing completed.");
                break;
            default:
                LOGGER.debug("Not supported input model specified.");
                break;
            }
        }
    }
    
    /**
     * Reads the source model and stores it in a {@link Project}.
     * @param in The input model, which shall be translated.
     * @param inType The type of <tt>in</tt>. Must be one of {@link InputType}.
     * @return The read model or <tt>null</tt> if an error occurred.
     */
    private static Project readSourceModel(File in, InputType inType) {
        Project model = null;

        LOGGER.info("1.) Read model."); 
        IModelReader reader = null;
        switch (inType) {
        case MODEL:
            try {
                reader = new ModelReader(in);
                model = reader.read();
            } catch (IOException e) {
                LOGGER.exception(e);
            } catch (ParserException e) {
                LOGGER.exception(e);
            }
            break;
        case RSF:
            try {
                reader = new RSFReader(in);
                model = reader.read();
            } catch (IOException e) {
                LOGGER.exception(e);
            } catch (ParserException e) {
                LOGGER.exception(e);
            }
            break;
        default:
            LOGGER.debug("Not supported input model specified.");
            break;
        }
        return model;
    }
    
    /**
     * Translates a model into another file format.
     * Default method which should be used.
     * @param in The input model, which shall be translated.
     * @param out The destination of the translated model.
     * @param inType The type of <tt>in</tt>. Must be one of {@link InputType}.
     * @param outType Specification in which format <tt>in</tt> shall be translated. Must be one of {@link OutputType}.
     * @param comment An optional comment (e.g. information about the translated model), can be <tt>null</tt>.
     * @param version Optionally the version of the translated model. Can be <tt>null</tt>. If not <tt>null</tt>,
     *     it must be in {@link de.uni_hildesheim.sse.utils.modelManagement.Version#Version(String)} syntax.
     */
    public static void translate(File in, File out, InputType inType, OutputType outType, String comment,
        String version) {
        
        try {
            FileWriter outWriter = new FileWriter(out);
            translate(in, outWriter, inType, outType, comment, version);
        } catch (IOException e) {
            LOGGER.exception(e);
        }
    }

    // checkstyle: resume parameter number check
}
