package iip.impl;

import java.io.InputStream;
import java.time.Clock;

import de.iip_ecosphere.platform.services.environment.ServiceKind;
import de.iip_ecosphere.platform.services.environment.YamlService;

import iip.datatypes.KRec13Anon;
import iip.datatypes.TurnstilePlcOutput;
import iip.impl.StateServiceImpl;

/**
 * IIP-Ecosphere service with state machine for service 'StateService'.
 * Generated by: EASy-Producer.
 */
public class StateService extends StateServiceImpl {

    private Turnstile fsm;
    private static StateService instance;

    /**
     * Fallback constructor setting most fields to "empty" default values.
     *
     * @param kind the service kind
     */
    protected StateService(ServiceKind kind) {
        super(kind);
        initiateStateMachine();
        instance = this;
    }

    /**
     * Fallback constructor setting most fields to "empty" default values.
     *
     * @param serviceId the id of the service
     * @param kind the service kind
     */
    protected StateService(String serviceId, ServiceKind kind) {
        super(serviceId, kind);
        initiateStateMachine();
        instance = this;
    }

    /**
     * Creates a service instance from YAML information.
     *
     * @param yaml the service information as read from YAML
     */
    protected StateService(YamlService yaml) {
        super(yaml);
        initiateStateMachine();
        instance = this;
    }

    /**
     * Creates a service instance from a service id and a YAML artifact.
     *
     * @param serviceId the id of the service
     * @param ymlFile the YML file containing the YAML artifact with the service descriptor
     */
    protected StateService(String serviceId, InputStream ymlFile) {
        super(serviceId, ymlFile);
        initiateStateMachine();
        instance = this;
    }

    @Override
    public void processTurnstilePlcOutput(TurnstilePlcOutput data) {
        // comment
        fsm.processTurnstilePlcOutput(data);
    }

    @Override
    public void processKRec13Anon(KRec13Anon data) {
        // comment
        fsm.processKRec13Anon(data);
    }

    public void initiateStateMachine() {
        this.fsm = new Turnstile(this, Clock.systemDefaultZone());
    }

    public interface State {

        String getName();

        void push();

        void coin();

    }

    protected abstract class AbstractStateMachine {

        private final Clock clock;
        private long lastStateChange;

        /**
         * 
         *
         * @param clock 
         */
        public AbstractStateMachine(Clock clock) {
            this.clock = clock;
            this.lastStateChange = now();
        }

        public long now() {
            return clock.millis();
        }

        public long millisSinceLastStateChange() {
            return now() - lastStateChange;
        }

        public boolean hasElapsed(long threshold) {
            return millisSinceLastStateChange() >= threshold;
        }

        public void markStateChange() {
            this.lastStateChange = now();
        }

    }

    protected class Turnstile extends AbstractStateMachine {

        private State currentState;
        private final StateService service;
        private final State lockedState;
        private final State unlockedState;

        public State getCurrentState() {
            return this.currentState;
        }

        public State getLockedState() {
            return lockedState;
        }

        public State getUnlockedState() {
            return unlockedState;
        }

        /**
         * 
         *
         * @param service 
         * @param clock 
         */
        public Turnstile(StateService service, Clock clock) {
            super(clock);
            this.lockedState = new LockedState(this);
            this.unlockedState = new UnlockedState(this);
            this.service = service;
            setState(lockedState);
        }

        public abstract class AbstractState implements State {

            private final Turnstile ctx;

            /**
             * 
             *
             * @param ctx 
             */
            public AbstractState(Turnstile ctx) {
                this.ctx = ctx;
            }

            @Override
            public void push() {
                logIgnored("push");
            }

            @Override
            public void coin() {
                logIgnored("coin");
            }

            void logIgnored(String transName) {
                System.out.println("[FSM] Transition " + transName + " is not defined in state "+ getName());
            }

            Turnstile getCtx() {
                return this.ctx;
            }

        }

        protected class LockedState extends AbstractState {

            /**
             * 
             *
             * @param ctx 
             */
            public LockedState(Turnstile ctx) {
                super(ctx);
            }

            public String getName() {
                return "locked";
            }

            @Override
            public void push() {
                getCtx().setState(getCtx().getUnlockedState());
            }

        }

        protected class UnlockedState extends AbstractState {

            /**
             * 
             *
             * @param ctx 
             */
            public UnlockedState(Turnstile ctx) {
                super(ctx);
            }

            public String getName() {
                return "unlocked";
            }

            @Override
            public void coin() {
                getCtx().setState(getCtx().getUnlockedState());
            }

        }

        public void setState(State state) {
            this.currentState = state;
            markStateChange();
        }

        public void processTurnstilePlcOutput(TurnstilePlcOutput data) {
            if (data.getReady()) {
                currentState.push();
            }
            if (data.getMoney() >= 3) {
                currentState.coin();
            }
        }

        public void processKRec13Anon(KRec13Anon data) {
        }

    }

}
