package iip.nodes;

import java.util.function.Consumer;
import javax.annotation.PostConstruct;

import de.iip_ecosphere.platform.services.environment.AbstractService;
import de.iip_ecosphere.platform.services.environment.MonitoringService;
import de.iip_ecosphere.platform.services.environment.spring.SpringAsyncServiceBase;
import de.iip_ecosphere.platform.services.environment.spring.Starter;
import de.iip_ecosphere.platform.services.environment.spring.metricsProvider.MetricsProvider;
import de.iip_ecosphere.platform.services.environment.switching.ServiceBase;
import de.iip_ecosphere.platform.support.logging.LoggerFactory;
import de.iip_ecosphere.platform.support.metrics.Counter;
import de.iip_ecosphere.platform.support.metrics.Timer;
import de.iip_ecosphere.platform.transport.Transport;

import iip.datatypes.Rec1;
import iip.interfaces.SimpleDataReceiverInterface;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.stream.function.StreamBridge;
import org.springframework.context.annotation.Bean;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.stereotype.Component;

/**
 * Spring Cloud Stream service frame for net node 'Simple Data Receiver'.
 * Generated by: EASy-Producer.
 */
@Component
@ConditionalOnProperty(value = "iip.service.SimpleReceiver", havingValue = "true", matchIfMissing = true)
@EnableScheduling
public class SimpleDataReceiver extends SpringAsyncServiceBase {

    @Value("${iip.service.SimpleReceiver:true}")
    private String activated;
    @Autowired
    private StreamBridge streamBridge;
    private SimpleDataReceiverInterface service;
    @Autowired
    private MetricsProvider metrics;
    private Counter serviceSent;
    private Counter serviceReceived;
    private Timer processingTime;
    private String appInstId = "";

    /**
     * Creates an instance.
     */
    public SimpleDataReceiver() {
        service = AbstractService.createInstance("de.iip_ecosphere.platform.test.apps.serviceImpl.SimpleReceiverImpl",
            SimpleDataReceiverInterface.class, "SimpleReceiver", "deployment.yml");
        appInstId = getAppInstIdSuffix(service, "_");
    }

    /**
     * Called when data arrived that shall be received (data sink).
     *
     * @return the data consumer functor
     */
    @Bean
    public Consumer<Rec1> receiveRec1_SimpleReceiver() {
        return data -> {
            MetricsProvider.increaseCounterBy(serviceReceived, 1.0);
            processingTime.record(() -> service.processRec1(data));
        };
    }

    /**
     * Initializes the service when feasible in Spring lifecycle.
     */
    @PostConstruct
    public void initService() {
        if (null == activated || "".equals(activated) || "true".equals(activated)) {
            LoggerFactory.getLogger(getClass())
                .info("Initializing service SimpleReceiver: {}", activated);
            String iId;
            String sId;
            sId = Starter.getServiceId(service);
            iId = ServiceBase.getApplicationInstanceId(sId);
            if (iId == null || iId.length() == 0) {
                iId = "dflt";
            }
            serviceSent = metrics.createServiceSentCounter("SimpleReceiver", sId, "SimpleMeshApp", iId);
            serviceReceived = metrics.createServiceReceivedCounter("SimpleReceiver", sId, "SimpleMeshApp", iId);
            processingTime = metrics.createServiceProcessingTimer("SimpleReceiver", sId, "SimpleMeshApp", iId);
            MonitoringService.setUp(service, metrics);
            Starter.mapService(service);
            service.attachFeedbackIngestor(data -> {
                MetricsProvider.increaseCounterBy(serviceSent, 1.0);
                Transport.send(c -> c.asyncSend("data_SimpleReceiver_Feedback_SimpleMeshApp" + appInstId, data), 
                    "SimpleReceiver", "processFeedback_SimpleSource-in-0");
            });
            createReceptionCallback("data_SimpleSource_Rec1_SimpleMeshApp" + appInstId, receiveRec1_SimpleReceiver(),
                Rec1.class, "receiveRec1_SimpleReceiver-in-0");
        }
    }

}
