package iip.nodes;

import java.util.HashMap;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.function.Supplier;
import javax.annotation.PostConstruct;

import de.iip_ecosphere.platform.services.environment.AbstractService;
import de.iip_ecosphere.platform.services.environment.MonitoringService;
import de.iip_ecosphere.platform.services.environment.ParameterConfigurer;
import de.iip_ecosphere.platform.services.environment.spring.SpringAsyncServiceBase;
import de.iip_ecosphere.platform.services.environment.spring.Starter;
import de.iip_ecosphere.platform.services.environment.spring.metricsProvider.MetricsProvider;
import de.iip_ecosphere.platform.services.environment.switching.ServiceBase;
import de.iip_ecosphere.platform.support.logging.LoggerFactory;
import de.iip_ecosphere.platform.support.metrics.Counter;
import de.iip_ecosphere.platform.support.metrics.Timer;
import de.iip_ecosphere.platform.transport.Transport;

import iip.datatypes.Feedback;
import iip.datatypes.Rec1;
import iip.interfaces.SimpleDataSourceInterface;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.stream.function.StreamBridge;
import org.springframework.context.annotation.Bean;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.stereotype.Component;

import reactor.core.publisher.Flux;

/**
 * Spring Cloud Stream service frame for net node 'Simple Data Source'.
 * Generated by: EASy-Producer.
 */
@Component
@ConditionalOnProperty(value = "iip.service.SimpleSource", havingValue = "true", matchIfMissing = true)
@EnableScheduling
public class SimpleDataSource extends SpringAsyncServiceBase {

    @Value("${iip.service.SimpleSource:true}")
    private String activated;
    @Autowired
    private StreamBridge streamBridge;
    private SimpleDataSourceInterface service;
    @Autowired
    private MetricsProvider metrics;
    private Counter serviceSent;
    private Counter serviceReceived;
    private Timer processingTime;
    private String appInstId = "";

    /**
     * Creates an instance.
     */
    public SimpleDataSource() {
        service = AbstractService.createInstance("de.iip_ecosphere.platform.test.apps.serviceImpl.SimpleSourceImpl",
            SimpleDataSourceInterface.class, "SimpleSource", "deployment.yml");
        HashMap<String, String> paramValues = new HashMap<>();
        ParameterConfigurer<?> cfg;
        cfg = service.getParameterConfigurer("camIP");
        if (null != cfg) {
            cfg.addValue(paramValues, "127.0.0.1");
        }
        cfg = service.getParameterConfigurer("camPort");
        if (null != cfg) {
            cfg.addValue(paramValues, -1);
        }
        try {
            service.reconfigure(paramValues);
        } catch (ExecutionException e) {
            LoggerFactory.getLogger(getClass())
                .error("Configuring initial parameter: {}", e.getMessage());
        }
        appInstId = getAppInstIdSuffix(service, "_");
    }

    /**
     * Called when data arrived that shall be processed (asynchronously).
     *
     * @return the data consumer functor
     */
    @Bean
    public Consumer<Feedback> processFeedback_SimpleSource() {
        return data -> {
            MetricsProvider.increaseCounterBy(serviceReceived, 1.0);
            processingTime.record(() -> service.processFeedback(data));
        };
    }

    /**
     * Creates data to be ingested.
     *
     * @return the data supplier functor
     */
    @Bean
    public Supplier<Flux<Rec1>> createRec1_SimpleSource() {
        // this is no real use of flux, but ensures that method is called only once to set the ingestor
        // when the mesh element is really active
        return () -> Flux.from(emitter -> {
            // empty emitter, else duplicates the effort of #28
        });
    }

    /**
     * Initializes the service when feasible in Spring lifecycle.
     */
    @PostConstruct
    public void initService() {
        if (null == activated || "".equals(activated) || "true".equals(activated)) {
            LoggerFactory.getLogger(getClass())
                .info("Initializing service SimpleSource: {}", activated);
            String iId;
            String sId;
            sId = Starter.getServiceId(service);
            iId = ServiceBase.getApplicationInstanceId(sId);
            if (iId == null || iId.length() == 0) {
                iId = "dflt";
            }
            serviceSent = metrics.createServiceSentCounter("SimpleSource", sId, "SimpleMeshApp", iId);
            serviceReceived = metrics.createServiceReceivedCounter("SimpleSource", sId, "SimpleMeshApp", iId);
            processingTime = metrics.createServiceProcessingTimer("SimpleSource", sId, "SimpleMeshApp", iId);
            MonitoringService.setUp(service, metrics);
            Starter.mapService(service);
            service.attachRec1Ingestor(data -> {
                MetricsProvider.increaseCounterBy(serviceSent, 1.0);
                Transport.send(c -> c.asyncSend("data_SimpleSource_Rec1_SimpleMeshApp" + appInstId, data), "SimpleSourc"
                    + "e", "receiveRec1_SimpleReceiver-in-0");
            });
            createReceptionCallback("data_SimpleReceiver_Feedback_SimpleMeshApp" + appInstId,
                processFeedback_SimpleSource(), Feedback.class, "processFeedback_SimpleSource-in-0");
        }
    }

}
