/*
 * Decompiled with CFR 0.152.
 */
package com.hivemq.mqtt.services;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.primitives.ImmutableIntArray;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import com.hivemq.bootstrap.ClientConnection;
import com.hivemq.bootstrap.ioc.lazysingleton.LazySingleton;
import com.hivemq.configuration.service.InternalConfigurations;
import com.hivemq.extension.sdk.api.annotations.NotNull;
import com.hivemq.extension.sdk.api.annotations.Nullable;
import com.hivemq.mqtt.callback.PublishStatusFutureCallback;
import com.hivemq.mqtt.handler.publish.PublishStatus;
import com.hivemq.mqtt.message.MessageWithID;
import com.hivemq.mqtt.message.QoS;
import com.hivemq.mqtt.message.dropping.MessageDroppedService;
import com.hivemq.mqtt.message.pool.FreePacketIdRanges;
import com.hivemq.mqtt.message.pool.exception.MessageIdUnavailableException;
import com.hivemq.mqtt.message.pool.exception.NoMessageIdAvailableException;
import com.hivemq.mqtt.message.publish.PUBLISH;
import com.hivemq.mqtt.message.publish.PUBLISHFactory;
import com.hivemq.mqtt.message.publish.PublishWithFuture;
import com.hivemq.mqtt.message.publish.PubrelWithFuture;
import com.hivemq.mqtt.message.pubrel.PUBREL;
import com.hivemq.mqtt.message.subscribe.Topic;
import com.hivemq.mqtt.services.PublishPollService;
import com.hivemq.mqtt.topic.SubscriberWithQoS;
import com.hivemq.persistence.SingleWriterService;
import com.hivemq.persistence.clientqueue.ClientQueuePersistence;
import com.hivemq.persistence.clientsession.ClientSessionSubscriptionPersistence;
import com.hivemq.persistence.clientsession.SharedSubscriptionService;
import com.hivemq.persistence.connection.ConnectionPersistence;
import com.hivemq.persistence.util.FutureUtils;
import com.hivemq.util.Exceptions;
import io.netty.channel.Channel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import javax.inject.Inject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@LazySingleton
public class PublishPollServiceImpl
implements PublishPollService {
    @NotNull
    private static final Logger log = LoggerFactory.getLogger(PublishPollService.class);
    @NotNull
    private final ClientQueuePersistence clientQueuePersistence;
    @NotNull
    private final ConnectionPersistence connectionPersistence;
    @NotNull
    private final MessageDroppedService messageDroppedService;
    @NotNull
    private final SharedSubscriptionService sharedSubscriptionService;
    @NotNull
    private final SingleWriterService singleWriterService;
    @NotNull
    private final ClientSessionSubscriptionPersistence clientSessionSubscriptionPersistence;

    @Inject
    public PublishPollServiceImpl(@NotNull ClientQueuePersistence clientQueuePersistence, @NotNull ConnectionPersistence connectionPersistence, @NotNull MessageDroppedService messageDroppedService, @NotNull SharedSubscriptionService sharedSubscriptionService, @NotNull SingleWriterService singleWriterService, @NotNull ClientSessionSubscriptionPersistence clientSessionSubscriptionPersistence) {
        this.clientQueuePersistence = clientQueuePersistence;
        this.connectionPersistence = connectionPersistence;
        this.messageDroppedService = messageDroppedService;
        this.sharedSubscriptionService = sharedSubscriptionService;
        this.singleWriterService = singleWriterService;
        this.clientSessionSubscriptionPersistence = clientSessionSubscriptionPersistence;
    }

    @Override
    public void pollMessages(@NotNull String client, @NotNull Channel channel) {
        Preconditions.checkNotNull((Object)client, (Object)"Client must not be null");
        Preconditions.checkNotNull((Object)channel, (Object)"Channel must not be null");
        ClientConnection clientConnection = ClientConnection.of(channel);
        boolean inflightMessagesSent = clientConnection.isInFlightMessagesSent();
        if (inflightMessagesSent) {
            this.pollNewMessages(client, channel);
            boolean noSharedSubscriptions = clientConnection.getNoSharedSubscription();
            if (noSharedSubscriptions) {
                return;
            }
            try {
                ImmutableSet<Topic> topics = this.sharedSubscriptionService.getSharedSubscriptions(client, () -> this.clientSessionSubscriptionPersistence.getSharedSubscriptions(client));
                if (topics.isEmpty()) {
                    clientConnection.setNoSharedSubscription(true);
                    return;
                }
                for (Topic topic : topics) {
                    String sharedSubscriptions = SharedSubscriptionService.removePrefix(topic.getTopic());
                    this.pollSharedPublishesForClient(client, sharedSubscriptions, topic.getQoS().getQosNumber(), topic.isRetainAsPublished(), topic.getSubscriptionIdentifier(), channel);
                }
            }
            catch (ExecutionException e) {
                log.error("Exception while reading shared subscriptions for client {}", (Object)client, (Object)e);
            }
        } else {
            this.pollInflightMessages(client, channel);
        }
    }

    @Override
    public void pollNewMessages(@NotNull String client) {
        ClientConnection clientConnection = this.connectionPersistence.get(client);
        if (clientConnection == null || clientConnection.getClientState().disconnectingOrDisconnected()) {
            return;
        }
        this.pollNewMessages(client, clientConnection.getChannel());
    }

    @Override
    public void pollNewMessages(final @NotNull String client, final @NotNull Channel channel) {
        ImmutableIntArray messageIds;
        final FreePacketIdRanges freePacketIdRanges = ClientConnection.of(channel).getFreePacketIdRanges();
        try {
            messageIds = this.createMessageIds(freePacketIdRanges, this.pollMessageLimit(channel));
        }
        catch (NoMessageIdAvailableException e) {
            log.error("No message id available for client {}", (Object)client, (Object)e);
            return;
        }
        ListenableFuture<ImmutableList<PUBLISH>> future = this.clientQueuePersistence.readNew(client, false, messageIds, 0x500000L);
        Futures.addCallback(future, (FutureCallback)new FutureCallback<ImmutableList<PUBLISH>>(){

            public void onSuccess(ImmutableList<PUBLISH> publishes) {
                int usedIds = 0;
                for (PUBLISH publish : publishes) {
                    if (publish.getQoS() == QoS.AT_MOST_ONCE) continue;
                    ++usedIds;
                }
                for (int i = usedIds; i < messageIds.length(); ++i) {
                    freePacketIdRanges.returnId(messageIds.get(i));
                }
                ArrayList<PublishWithFuture> publishesToSend = new ArrayList<PublishWithFuture>(publishes.size());
                AtomicInteger inFlightMessageCount = PublishPollServiceImpl.this.inFlightMessageCount(channel);
                inFlightMessageCount.addAndGet(publishes.size());
                for (PUBLISH publish : publishes) {
                    SettableFuture publishFuture = SettableFuture.create();
                    Futures.addCallback((ListenableFuture)publishFuture, (FutureCallback)new PublishStatusFutureCallback(PublishPollServiceImpl.this, false, client, publish, freePacketIdRanges, channel, client), (Executor)MoreExecutors.directExecutor());
                    PublishWithFuture publishWithFuture = new PublishWithFuture(publish, (SettableFuture<PublishStatus>)publishFuture, false);
                    publishesToSend.add(publishWithFuture);
                }
                ClientConnection.of(channel).getPublishFlushHandler().sendPublishes(publishesToSend);
            }

            public void onFailure(Throwable t) {
                Exceptions.rethrowError("Exception in new messages handling", t);
                channel.disconnect();
            }
        }, (Executor)this.singleWriterService.callbackExecutor(client));
    }

    @Override
    public void pollInflightMessages(final @NotNull String client, final @NotNull Channel channel) {
        ListenableFuture<ImmutableList<MessageWithID>> future = this.clientQueuePersistence.readInflight(client, 0x500000L, this.pollMessageLimit(channel));
        Futures.addCallback(future, (FutureCallback)new FutureCallback<ImmutableList<MessageWithID>>(){

            public void onSuccess(ImmutableList<MessageWithID> messages) {
                ClientConnection clientConnection = ClientConnection.of(channel);
                if (messages.isEmpty()) {
                    clientConnection.setInFlightMessagesSent(true);
                    channel.eventLoop().submit(() -> PublishPollServiceImpl.this.pollMessages(client, channel));
                    return;
                }
                ArrayList<PublishWithFuture> publishesToSend = new ArrayList<PublishWithFuture>(messages.size());
                AtomicInteger inFlightMessageCount = PublishPollServiceImpl.this.inFlightMessageCount(channel);
                inFlightMessageCount.addAndGet(messages.size());
                int messagesSize = messages.size();
                for (int i = 0; i < messagesSize; ++i) {
                    MessageWithID message = (MessageWithID)messages.get(i);
                    FreePacketIdRanges freePacketIdRanges = clientConnection.getFreePacketIdRanges();
                    try {
                        freePacketIdRanges.takeSpecificId(message.getPacketIdentifier());
                    }
                    catch (MessageIdUnavailableException e) {
                        log.warn("The desired packet ID was not available when polling inflight messages: {}", (Object)e.getMessage());
                    }
                    if (message instanceof PUBLISH) {
                        PUBLISH publish = (PUBLISH)message;
                        SettableFuture publishFuture = SettableFuture.create();
                        Futures.addCallback((ListenableFuture)publishFuture, (FutureCallback)new PublishStatusFutureCallback(PublishPollServiceImpl.this, false, client, publish, freePacketIdRanges, channel, client), (Executor)MoreExecutors.directExecutor());
                        PublishWithFuture publishWithFuture = new PublishWithFuture(publish, (SettableFuture<PublishStatus>)publishFuture, false);
                        publishesToSend.add(publishWithFuture);
                        continue;
                    }
                    if (!(message instanceof PUBREL)) continue;
                    SettableFuture settableFuture = SettableFuture.create();
                    channel.writeAndFlush((Object)new PubrelWithFuture((PUBREL)message, (SettableFuture<PublishStatus>)settableFuture));
                    Futures.addCallback((ListenableFuture)settableFuture, (FutureCallback)new PubrelResendCallback(client, message, freePacketIdRanges, channel), (Executor)MoreExecutors.directExecutor());
                }
                clientConnection.getPublishFlushHandler().sendPublishes(publishesToSend);
            }

            public void onFailure(Throwable t) {
                Exceptions.rethrowError("Exception in inflight messages handling", t);
            }
        }, (Executor)this.singleWriterService.callbackExecutor(client));
    }

    @NotNull
    private AtomicInteger inFlightMessageCount(@NotNull Channel channel) {
        ClientConnection clientConnection = ClientConnection.of(channel);
        if (clientConnection.getInFlightMessageCount() == null) {
            clientConnection.setInFlightMessageCount(new AtomicInteger(0));
        }
        return clientConnection.getInFlightMessageCount();
    }

    @Override
    public void pollSharedPublishes(@NotNull String sharedSubscription) {
        ArrayList<SubscriberWithQoS> subscribers = new ArrayList<SubscriberWithQoS>((Collection<SubscriberWithQoS>)this.sharedSubscriptionService.getSharedSubscriber(sharedSubscription));
        for (int backIndex = subscribers.size(); backIndex > 0; --backIndex) {
            int chosenIndex = ThreadLocalRandom.current().nextInt(backIndex);
            SubscriberWithQoS subscriber = (SubscriberWithQoS)subscribers.get(chosenIndex);
            ClientConnection clientConnection = this.connectionPersistence.get(subscriber.getSubscriber());
            if (clientConnection == null || !clientConnection.getChannel().isActive()) {
                SubscriberWithQoS backSubscriber = (SubscriberWithQoS)subscribers.get(backIndex - 1);
                subscribers.set(chosenIndex, backSubscriber);
                continue;
            }
            this.pollSharedPublishesForClient(subscriber.getSubscriber(), sharedSubscription, subscriber.getQos(), subscriber.isRetainAsPublished(), subscriber.getSubscriptionIdentifier(), clientConnection.getChannel());
        }
    }

    @Override
    public void pollSharedPublishesForClient(final @NotNull String client, final @NotNull String sharedSubscription, final int qos, final boolean retainAsPublished, final @Nullable Integer subscriptionIdentifier, final @NotNull Channel channel) {
        final ClientConnection clientConnection = ClientConnection.of(channel);
        if (clientConnection.isMessagesInFlight()) {
            return;
        }
        ListenableFuture<ImmutableList<PUBLISH>> future = this.clientQueuePersistence.readShared(sharedSubscription, this.pollMessageLimit(channel), 0x500000L);
        Futures.addCallback(future, (FutureCallback)new FutureCallback<ImmutableList<PUBLISH>>(){

            public void onSuccess(@NotNull ImmutableList<PUBLISH> publishes) {
                if (publishes.isEmpty()) {
                    return;
                }
                FreePacketIdRanges freePacketIdRanges = clientConnection.getFreePacketIdRanges();
                ArrayList<PublishWithFuture> publishesToSend = new ArrayList<PublishWithFuture>(publishes.size());
                AtomicInteger inFlightMessageCount = PublishPollServiceImpl.this.inFlightMessageCount(channel);
                inFlightMessageCount.addAndGet(publishes.size());
                for (PUBLISH publish : publishes) {
                    if (publish.getOnwardQoS().getQosNumber() > 0 && qos == 0) {
                        PublishPollServiceImpl.this.removeMessageFromSharedQueue(sharedSubscription, publish.getUniqueId());
                    }
                    QoS minQos = QoS.valueOf(Math.min(qos, publish.getOnwardQoS().getQosNumber()));
                    ImmutableIntArray subscriptionIdentifiers = subscriptionIdentifier != null ? ImmutableIntArray.of((int)subscriptionIdentifier) : ImmutableIntArray.of();
                    int packetId = 0;
                    try {
                        if (((QoS)((Object)Preconditions.checkNotNull((Object)((Object)minQos)))).getQosNumber() > 0) {
                            packetId = freePacketIdRanges.takeNextId();
                        }
                    }
                    catch (NoMessageIdAvailableException e) {
                        log.error("No message id available for client: {}, shared subscription {}", (Object)client, (Object)sharedSubscription);
                        PublishPollServiceImpl.this.messageDroppedService.queueFullShared(sharedSubscription, publish.getTopic(), publish.getQoS().getQosNumber());
                        inFlightMessageCount.decrementAndGet();
                        return;
                    }
                    PUBLISH publishToSend = new PUBLISHFactory.Mqtt5Builder().fromPublish(publish).withPacketIdentifier(packetId).withQoS(minQos).withOnwardQos(minQos).withRetain(publish.isRetain() && retainAsPublished).withSubscriptionIdentifiers(subscriptionIdentifiers).build();
                    SettableFuture publishFuture = SettableFuture.create();
                    Futures.addCallback((ListenableFuture)publishFuture, (FutureCallback)new PublishStatusFutureCallback(PublishPollServiceImpl.this, true, sharedSubscription, publishToSend, freePacketIdRanges, channel, client), (Executor)MoreExecutors.directExecutor());
                    PublishWithFuture publishWithFuture = new PublishWithFuture(publishToSend, (SettableFuture<PublishStatus>)publishFuture, false);
                    publishesToSend.add(publishWithFuture);
                }
                clientConnection.getPublishFlushHandler().sendPublishes(publishesToSend);
            }

            public void onFailure(@NotNull Throwable t) {
                Exceptions.rethrowError("Exception in shared publishes poll handling for client " + client + "for shared subscription " + sharedSubscription, t);
            }
        }, (Executor)this.singleWriterService.callbackExecutor(client));
    }

    @Override
    @NotNull
    public ListenableFuture<Void> removeMessageFromQueue(@NotNull String client, int packetId) {
        return this.clientQueuePersistence.remove(client, packetId);
    }

    @Override
    @NotNull
    public ListenableFuture<Void> removeMessageFromSharedQueue(@NotNull String sharedSubscription, @NotNull String uniqueId) {
        return this.clientQueuePersistence.removeShared(sharedSubscription, uniqueId);
    }

    @Override
    @NotNull
    public ListenableFuture<Void> putPubrelInQueue(@NotNull String client, int packetId) {
        return this.clientQueuePersistence.putPubrel(client, packetId);
    }

    @Override
    @NotNull
    public ListenableFuture<Void> removeInflightMarker(@NotNull String sharedSubscription, @NotNull String uniqueId) {
        return this.clientQueuePersistence.removeInFlightMarker(sharedSubscription, uniqueId);
    }

    @NotNull
    private ImmutableIntArray createMessageIds(@NotNull FreePacketIdRanges messageIDPool, int pollMessageLimit) throws NoMessageIdAvailableException {
        ImmutableIntArray.Builder builder = ImmutableIntArray.builder((int)pollMessageLimit);
        for (int i = 0; i < pollMessageLimit; ++i) {
            int nextId = messageIDPool.takeNextId();
            builder.add(nextId);
        }
        return builder.build();
    }

    private int pollMessageLimit(@NotNull Channel channel) {
        ClientConnection clientConnection = ClientConnection.of(channel);
        int maxInflightWindow = clientConnection.getMaxInflightWindow(InternalConfigurations.MAX_INFLIGHT_WINDOW_SIZE_MESSAGES);
        return Math.max(InternalConfigurations.PUBLISH_POLL_BATCH_SIZE, maxInflightWindow);
    }

    private class PubrelResendCallback
    implements FutureCallback<PublishStatus> {
        @NotNull
        private final String client;
        @NotNull
        private final MessageWithID message;
        @NotNull
        private final FreePacketIdRanges messageIDPool;
        @NotNull
        private final Channel channel;

        PubrelResendCallback(@NotNull String client, @NotNull MessageWithID message, @NotNull FreePacketIdRanges messageIDPool, Channel channel) {
            this.client = client;
            this.message = message;
            this.messageIDPool = messageIDPool;
            this.channel = channel;
        }

        public void onSuccess(@NotNull PublishStatus result) {
            AtomicInteger inFlightMessages;
            this.messageIDPool.returnId(this.message.getPacketIdentifier());
            if (result != PublishStatus.NOT_CONNECTED) {
                ListenableFuture<Void> future = PublishPollServiceImpl.this.removeMessageFromQueue(this.client, this.message.getPacketIdentifier());
                FutureUtils.addExceptionLogger(future);
            }
            if ((inFlightMessages = ClientConnection.of(this.channel).getInFlightMessageCount()) != null && inFlightMessages.decrementAndGet() > 0) {
                return;
            }
            PublishPollServiceImpl.this.pollMessages(this.client, this.channel);
        }

        public void onFailure(Throwable t) {
            Exceptions.rethrowError("Pubrel delivery failed", t);
            this.messageIDPool.returnId(this.message.getPacketIdentifier());
            AtomicInteger inFlightMessages = ClientConnection.of(this.channel).getInFlightMessageCount();
            if (inFlightMessages != null) {
                inFlightMessages.decrementAndGet();
            }
        }
    }
}

