package gr.tuc.softnet.ap0n.graph;

import java.io.Serializable;

import gr.tuc.softnet.ap0n.utils.Interval;

/**
 * Created by Apostolos Nydriotis on 2014/11/25.
 */
public class Edge implements Comparable<Edge>, Serializable {

  private static final long serialVersionUID = 6993252545069720898L;
  private String idA;
  private String idB;
  private long timestamp;  // Generically used. should sort it out a bit
  private Interval lifetime;

  /**
   * Constructs an *Undirected* edge
   * @param idA
   * @param idB
   * @param timestamp
   */
  public Edge(String idA, String idB, long timestamp) {
    // Use lexicographic ordering for the ids
    if (idA.compareTo(idB) < 0) {
      this.idA = idA;
      this.idB = idB;
    } else {
      this.idA = idB;
      this.idB = idA;
    }
    this.timestamp = timestamp;
    this.lifetime = new Interval(timestamp, null);
  }

  public Edge(Edge o) {
    this.idA = o.idA;
    this.idB = o.idB;
    this.timestamp = o.timestamp;
    this.lifetime = o.lifetime;
  }

  public String getIdA() {
    return idA;
  }

  public String getIdB() {
    return idB;
  }

  public String getId() { return idA + "," + idB; }

  public long getTimestamp() {
    return timestamp;
  }

  public void setTimestamp(long timestamp) {
    this.timestamp = timestamp;
  }

  public Interval getLifetime() {
    return lifetime;
  }

  public void setLifetime(Interval lifetime) {
    this.lifetime = lifetime;
  }

  /**
   *
   * @param id
   * @return true if idA or idB is id (independently of the time)
   */
  public boolean reachesNode(String id) {
    return id.equals(idA) || id.equals(idB);
  }

  @Override
  public String toString() {
    return "(" + timestamp + ":" + idA + "," + idB + ")";
  }

  /**
   * Compare this edge to another edge
   *
   * @param other edge to compare to this edge
   * @return 0 if this edge is equal to other, -1/+1 otherwise
   */
  public int compareTo(Edge other) {

    // appearance timestamp comparison
    Long thisStart = this.lifetime.getStart();
    Long otherStart = other.lifetime.getStart();
    if (thisStart < otherStart) {
      return -1;
    } else if (thisStart > otherStart) {
      return 1;
    }

    // Id lexicographical comparison
    String thisId = this.getId();
    String otherId = other.getId();
    if (thisId.compareTo(otherId) < 0) {
      return -1;
    } else if (thisId.compareTo(otherId) > 0) {
      return 1;
    }

    // Disappearance timestamp comparison
    Long thisEnd = this.lifetime.getEnd();
    Long otherEnd = other.lifetime.getEnd();
    if (thisEnd == null && otherEnd == null) {
      return 0;
    } else if (thisEnd != null && otherEnd == null) {
      return -1;
    } else if (thisEnd == null && otherEnd != null) {
      return 1;
    } else if (thisEnd < otherEnd) {
      return -1;
    } else if (thisEnd > otherEnd) {
      return 1;
    }
    return 0;
  }
}
