package gr.tuc.softnet.ap0n.index;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;

import gr.tuc.softnet.ap0n.graph.Edge;
import gr.tuc.softnet.ap0n.graph.Graph;
import gr.tuc.softnet.ap0n.graph.Snapshot;
import gr.tuc.softnet.ap0n.graph.Vertex;
import gr.tuc.softnet.ap0n.index.imp.IEEdgeNode;
import gr.tuc.softnet.ap0n.index.imp.INEdgeNode;
import gr.tuc.softnet.ap0n.index.imp.INVersionNode;
import gr.tuc.softnet.ap0n.index.inf.INNode;
import gr.tuc.softnet.ap0n.persistence.imp.MemmoryDataAccessor;
import gr.tuc.softnet.ap0n.persistence.inf.DataAccessor;
import gr.tuc.softnet.ap0n.utils.DEntry;
import gr.tuc.softnet.ap0n.utils.Interval;
import gr.tuc.softnet.ap0n.utils.PathQueryType;

/**
 * Created by Apostolos Nydriotis on 2014/11/24.
 */
public class VolatileIndex {

  private DataAccessor dataAccessor;  // Provides access to I-N (persisted)
  private ConcurrentSkipListMap<Long, IEEdgeNode> ie;  // I-E skiplist (in-memory at least for now)

  public VolatileIndex() throws Exception {
    dataAccessor = new MemmoryDataAccessor();
//    dataAccessor = new BerkeleyDBDataAccessor("./db");
//    dataAccessor = new InfinispanDataAccessor();
    dataAccessor.initialize();
    ie = new ConcurrentSkipListMap<Long, IEEdgeNode>();
  }

  /**
   * Adds the vertex to the index. vertex.timestamp is the appearance timestamp
   */
  public void addGraphNode(Vertex vertex) throws Exception {
    addVersionNode(vertex, false);
  }

  /**
   * Marks the vertex as DELETED. vertex.timestamp is the expiration timestamp
   */
  public void expireGraphNode(Vertex vertex) throws Exception {
    addVersionNode(vertex, true);
  }

  /**
   * Adds the edge to the gr.tuc.softnet.ap0n.index. edge.timestamp is the appearance timestamp
   */
  public void addGraphEdge(Edge edge) throws Exception {
    addEdgeNode(edge, false);
  }

  /**
   * Marks the edge as DELETED. edge.timestamp is the expiration timestamp
   */
  public void expireEdge(Edge edge) throws Exception {
    addEdgeNode(edge, true);
  }

  public VolatileIndexKey getIndexKey(String id) throws Exception {
    return dataAccessor.fetchIndexKey(id);
  }

  public List<VolatileIndexKey> getKeys() throws Exception {
    return dataAccessor.fetchIndex();
  }

  public Vertex getFirstVersionNode(String id) throws Exception {
    VolatileIndexKey key = getIndexKey(id);
    for (INNode n : key.getNodes()) {
      if (n.getType() == IndexNodeType.VERSION) {
        return ((INVersionNode) n).getVertex();
      }
    }
    return null;
  }

  public void close() throws Exception {
    dataAccessor.shutdown();
  }

  @Deprecated
  public List<Snapshot> getSnapshots(long ts, long te) throws Exception {

    List<Snapshot> snapshots = new ArrayList<Snapshot>();

    for (Long t = ts; t <= te; t++) {
      snapshots.add(getSnapshot(t));
    }

    return snapshots;
  }

  public Snapshot getSnapshot(Long timestamp) throws Exception {
    Graph graph = new Graph();
    Snapshot snapshot = new Snapshot(timestamp, graph);

    // Add the nodes
    for (Vertex n : getAliveNodes(timestamp)) {
      graph.addNode(n);
    }

    // Add the edges
    for (Edge e : getAliveEdges(new Interval(timestamp, timestamp))) {
      graph.addEdge(e);
    }

    return snapshot;
  }

  /**
   * Returns the edges that are alive into the interval.
   */
  public Set<Edge> getAliveEdges(Interval interval) {
    Set<Edge> result = new HashSet<>();

    // TODO: Maybe we should add somekind of check-pointing (to avoid going all the way down to
    // the beginning of the edgeList).
    // Get a view (sub-map) of the skiplist [begining, interval.end]
    ConcurrentNavigableMap<Long, IEEdgeNode> subMap;
    if (interval.getEnd() == null) {
      subMap = ie;
    } else {
      if (ie.firstKey() >= interval.getEnd()) {
        System.err.println(ie.firstKey() + " > " + interval.getEnd());
        return result;
      }
      subMap = ie.subMap(ie.firstKey(), true, interval.getEnd(), true);
    }

    for (Map.Entry<Long, IEEdgeNode> entry : subMap.descendingMap().entrySet()) {
      long timestamp = entry.getKey();
      IEEdgeNode element = entry.getValue();

      if (element.getTimestamp() < interval.getEnd() ||
          (interval.isTimepoint() && element.getTimestamp() == interval.getStart())) {
        // interval is [t1, t2) OR [t3,t3) for "timepoints"
        // Add the unbounded edges
        for (Edge e : element.getUnbounded().values()) {
          result.add(new Edge(e));
        }
        if (timestamp >= interval.getStart()) {
          for (Edge e : element.getBounded().values()) {
            result.add(new Edge(e));
          }
        } else {
          for (Edge e : element.getBounded().values()) {
            // e.getTimestamp() > (not >=) because the edge expired at e.getTimestamp. Therefore it
            // doesn't exist at e.getTimestamp
            if (e.getTimestamp() > interval.getStart()) {
              // Add the edge to the entry only if was deleted after the beginning of the interval.
              result.add(new Edge(e));
            }
          }
        }
      }
    }
    return result;
  }

  public DEntry processPathQuery(Interval query_interval, Vertex na, Vertex nb, PathQueryType type)
      throws Exception {
    QueryExecutor qe = new QueryExecutor(this, query_interval, na, nb, type);
    return qe.execute();
  }

  public void printToFile(String path) throws IOException {
    File file = new File(path);
    if (file.exists()) {
      file.delete();
    }
    file.createNewFile();
    FileWriter fw = new FileWriter(file, false);
    BufferedWriter bw = new BufferedWriter(fw);

    bw.write("* I-N\n");
    try {
      for (VolatileIndexKey key : dataAccessor.fetchIndex()) {
        bw.write(key.getId() + ": ");
        for (INNode node : key.getNodes()) {
          bw.write(node.toString() + " ");
        }
        bw.newLine();
      }
    } catch (Exception e) {
      e.printStackTrace();
    }
    bw.newLine();
    bw.write("* I-E\n");
    for (IEEdgeNode edgeNode : ie.values()) {
      bw.write(edgeNode.toString());
    }
    bw.close();
  }

  /**
   * Way too heavy!
   */
  @Override
  public String toString() {
    StringBuilder s = new StringBuilder();
    s.append("* I-N\n");
    try {
      for (VolatileIndexKey key : dataAccessor.fetchIndex()) {
        s.append(key.getId()).append(": ");
        for (INNode node : key.getNodes()) {
          s.append(node.toString()).append(" ");
        }
        s.append("\n");
      }
    } catch (Exception e) {
      e.printStackTrace();
    }
    s.append("\n");
    s.append("* I-E\n");
    for (IEEdgeNode edgeNode : ie.values()) {
      s.append(edgeNode.toString());
    }
    return s.toString();
  }

  public List<VolatileIndexKey> getIndex() throws Exception {
    return dataAccessor.fetchIndex();
  }

  public void clear() throws Exception {
    dataAccessor.clear();
  }

  private void addVersionNode(Vertex vertex, boolean isDeletion)
      throws Exception {
    VolatileIndexKey key = dataAccessor.fetchIndexKey(vertex.getId());

    if (isDeletion) {
      assert (key != null);
    }

    INVersionNode newVersionNode = new INVersionNode();
    newVersionNode.setVertex(vertex);
    newVersionNode.setDeletion(isDeletion);

    Set<String> edgesToDelete = new HashSet<String>();
    if (isDeletion) {
      // Reverse-iterate the key's queue. Delete every not deleted edge of the vertex.
      Iterator<INNode> iter = key.getNodes().descendingIterator();
      Set<String> deletedEdges = new HashSet<String>();

      while (iter.hasNext()) {
        INNode indexNode = (INNode) iter.next();
        if (indexNode.getType() != IndexNodeType.EDGE) {  // Don't process version nodes
          continue;
        }
        INEdgeNode edgeNode = (INEdgeNode) indexNode;
        if (deletedEdges.contains(edgeNode.getId())) {
          // Edge has already been deleted
          continue;
        } else if (edgeNode.getOperation() == EdgeOperation.DELETION) {
          // Edge is deleted. Note it.
          deletedEdges.add(edgeNode.getId());
          continue;
        } else {
          // Mark edges for deletion
          edgesToDelete.add(edgeNode.getId());
          deletedEdges.add(edgeNode.getId());
        }
      }
    }

    if (key == null) {
      // Vertex doesn't exist in the index. Create it.
      key = new VolatileIndexKey();
      key.setId(vertex.getId());
      key.setTimestamp(vertex.getTimestamp());
    }

    key.addNode(newVersionNode);
    dataAccessor.storeIndexKey(key);

    for (String nodeId : edgesToDelete) {
      // Delete the marked edges here to avoid concurrent modification exception and avoid
      // overwriting the index key (that would happen if we call this method before calling
      // the dataAccessor.storeIndexKey(key);
      addEdgeNode(new Edge(vertex.getId(), nodeId, vertex.getTimestamp()), true);
    }
  }

  /**
   * TODO: *** Try adding an existing edge ***
   */
  private void addEdgeNode(Edge edge, boolean isDeletion) throws Exception {
    VolatileIndexKey nodeAKey = dataAccessor.fetchIndexKey(edge.getIdA());
    VolatileIndexKey nodeBKey = dataAccessor.fetchIndexKey(edge.getIdB());

    if (nodeAKey == null) {
      throw new RuntimeException(edge.getIdA() + " node is not in the index");
    } else if (nodeBKey == null) {
      throw new RuntimeException(edge.getIdB() + " node is not in the index");
    }

    // Update I-E
    if (isDeletion) {
      // Expire the edge
      List<Long> appearanceTimestamps = getEdgeAppearanceTimestamps(edge);
      if (appearanceTimestamps.size() == 0) {
        // Edge not found in I-N
        throw new RuntimeException("Trying to delete non-existing edge " + edge.toString());
      }
      Long lastAppearanceTimestamp = appearanceTimestamps.get(appearanceTimestamps.size() - 1);
      if (lastAppearanceTimestamp > edge.getTimestamp()) {
        throw new RuntimeException(
            "Insertion timestamp > Deletion timestamp!(" + lastAppearanceTimestamp + " > " + edge
                .getTimestamp() + ")");
      }
      IEEdgeNode hostElement = ie.get(lastAppearanceTimestamp);
      hostElement.expireEdge(edge);
    } else {
      // Add the new edge
      if (!nodeAKey.isAlive(edge.getTimestamp()) || !nodeBKey.isAlive(edge.getTimestamp())) {
        throw new RuntimeException("Trying to add edge to non-existing node! " + edge.toString());
      }
      IEEdgeNode hostElement = ie.get(edge.getLifetime().getStart());
      if (hostElement == null) {
        hostElement = new IEEdgeNode(edge.getLifetime().getStart());
        ie.put(hostElement.getTimestamp(), hostElement);
      }
      hostElement.addUnboundedEdge(new Edge(edge));
    }

    // Update I-N: Add the new edge

    EdgeOperation operation = isDeletion ? EdgeOperation.DELETION : EdgeOperation.APPEARANCE;
    INEdgeNode edgeForA = new INEdgeNode(edge.getIdB(),
                                         operation,
                                         edge.getLifetime().getStart());
    nodeAKey.addNode(edgeForA);

    INEdgeNode edgeForB = new INEdgeNode(edge.getIdA(),
                                         operation,
                                         edge.getLifetime().getStart());
    nodeBKey.addNode(edgeForB);

    dataAccessor.storeIndexKey(nodeAKey);
    dataAccessor.storeIndexKey(nodeBKey);
  }

  /**
   * Uses the I-N to get the list of appearance timestamps of the edge.
   */
  private List<Long> getEdgeAppearanceTimestamps(Edge edge) {
    List<Long> result = new LinkedList<>();
    VolatileIndexKey key = null;
    try {
      key = dataAccessor.fetchIndexKey(edge.getIdA());
    } catch (Exception e) {
      e.printStackTrace();
    }
    for (INNode node : key.getNodes()) {
      if (node.getType() == IndexNodeType.EDGE) {
        INEdgeNode edgeNode = (INEdgeNode) node;
        if (edgeNode.getId().equals(edge.getIdB())) {
          if (edgeNode.getOperation() == EdgeOperation.APPEARANCE) {
            // The edge is found. However since it can be deleted and appear again, we must search
            // the whole node in order to contains the latest appearance.
            result.add(node.getTimestamp());
          }
        }
      }
    }
    return result;
  }

  private Set<Vertex> getAliveNodes(Long timestamp) throws Exception {
    Set<Vertex> result = new HashSet<>();

    for (VolatileIndexKey key : getKeys()) {
      // TODO: Keep a boolean at the key to know if it is alive or not (keys on the HD?)
      if (key.getTimestamp() > timestamp) {
        continue;  // We don't care about these timestamps
      }
      Iterator<INNode> nodes = key.getNodes().descendingIterator();
      while (nodes.hasNext()) {
        INNode node = nodes.next();
        if (node.getTimestamp() > timestamp || node.getType() != IndexNodeType.VERSION) {
          // We don't care about edges right now. We also skip versions out of the interval
          continue;
        }
        INVersionNode version = (INVersionNode) node;
        if (!version.isDeletion()) {
          // here version.getTimestamp() <= timestamp. Therefore, if the operation is not deletion,
          // the node is alive and this is its version.
          result.add(new Vertex(version.getVertex()));
        }  // Else the node is not alive.
        break;  // No need to iterate over the remaining nodes.
      }
    }

    return result;
  }

  /**
   * Returns the intervals where the vertex is alive
   *
   * @throws Exception
   */
  public List<Interval> getNodeLifetimes(Vertex vertex) throws Exception {
    List<Interval> result = new ArrayList<Interval>();
    Long start = null;
    VolatileIndexKey key = getIndexKey(vertex.getId());
    for (INNode n : key.getNodes()) {
      if (n.getType() == IndexNodeType.VERSION) {
        INVersionNode version = (INVersionNode) n;
        if (version.isDeletion()) {
          if (start == null) {
            throw new RuntimeException("Vertex disappeared with out having appeared!");
          }
          result.add(new Interval(start, version.getTimestamp()));
          start = null;
        } else if (start == null) {  // Else it's just an edit
          start = version.getTimestamp();
        }
      }
    }

    if (start != null) {
      result.add(new Interval(start, null));
    }
    return result;
  }
}
