package gr.tuc.softnet.ap0n.utils;

import java.io.Serializable;
import java.util.List;

/**
 * Represents the interval [start, end)
 * null represents infinity
 *
 * Created by ap0n on 17/5/2016.
 */
public class Interval implements Serializable {

  private static final long serialVersionUID = 1740368951734670777L;
  private Long start;
  private Long end;

  public Interval(Interval o) {
    this.start = o.start;
    this.end = o.end;
  }

  public Interval() {
    start = null;
    end = null;
  }

  public Interval(Long start, Long end) {
    if (start == null || end == null) {
      this.start = start;
      this.end = end;
    } else {
      this.start = start <= end ? start : end;
      this.end = end > start ? end : start;
    }
  }

  public Interval(int start, int end) {
    this((long) start, (long) end);
  }

  public static Interval getIntersection(Interval a, Interval b) {

    if (a.start == null && a.end == null) {
      return new Interval(b);
    } else if (b.start == null && b.end == null) {
      return new Interval(a);
    }

    if ((a.end != null && b.start != null && a.end < b.start) ||
        (a.start != null && b.end != null && a.start > b.end)) {
      //       |---| |---|
      // -Inf ------------- +Inf
      return null;
    }

    Interval result = new Interval();

    if ((a.start != null && b.start != null && a.start >= b.start) ||
        b.start == null) {
      //       ||---
      //       |----
      // -Inf ------- +Inf
      result.start = a.start;
    } else {
      //       |----
      //        |---
      // -Inf ------- +Inf
      result.start = b.start;
    }

    if ((a.end != null && b.end != null && a.end <= b.end) ||
        b.end == null) {
      //       ---||
      //       ----|
      // -Inf ------- +Inf
      result.end = a.end;
    } else {
      //       ----|
      //       ---|
      // -Inf ------- +Inf
      result.end = b.end;
    }
    return result;
  }

  public static Interval getIntersection(List<Interval> intervals) {
    if (intervals.size() == 0) {
      return null;
    }

    Interval intersection = intervals.get(0);

    for (Interval interval : intervals) {
      if (intersection == null) {
        break;
      }
      intersection = getIntersection(intersection, interval);
    }
    return intersection;
  }

  /**
   * @return true if this interval contains o
   */
  public boolean contains(Interval o) {
    if ((o.start >= start || start == null) && (o.end <= end || end == null)) {
      return true;
    }
    return false;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof Interval)) {
      return false;
    }
    Interval other = (Interval) o;
    boolean starts_are_equal = false;
    if ((other.getStart() == null && this.getStart() == null)
        || (other.getStart() != null && this.getStart() != null
            && other.getStart().equals(this.getStart()))) {
      starts_are_equal = true;
    }

    boolean ends_are_equal = false;
    if ((other.getEnd() == null && this.getEnd() == null)
        || (other.getEnd() != null && this.getEnd() != null && other.getEnd()
        .equals(this.getEnd()))) {
      ends_are_equal = true;
    }

    return starts_are_equal && ends_are_equal;
  }

  @Override
  public String toString() {
    return "[" + start + "," + end + ")";
  }

  public Long getStart() {
    return start;
  }

  public Long getEnd() {
    return end;
  }

  public void setEnd(Long end) {
    this.end = end;
  }

  public boolean isTimepoint() {
    return start != null && end != null && start == end;
  }
}
